/* reformat.p
 * Reads a named file, reformats it and writes it out to
 * a named file.
 *
 * The purpose of this procedure, for now, is not really to write
 * nice looking code, but just to write code in a consistent format
 * so that a diff utility will be able to find changes to code logic,
 * rather than find changes to code formatting.
 * For now, there isn't even an option for displaying comments.
 *
 * Initial version by John Green, Joanju Limited, April 2003.
 *
 * History:
 *
 * Date       Name        Description
 *
 * 03/27/06   G Campbell  Added code to correctly handle the indentation of
 *                        TRIGGERS sections.  Also, added output to PDF.  This
 *                        gives the ability to 'view' the blocks.  To disable
 *                        PDF output, remove preprocessor format2PDF
 */

/* igc - Remove the following line if you don't want PDF output */
&GLOBAL-DEFINE format2PDF


define input parameter sourceFileName as character no-undo.
define input parameter targetFileName as character no-undo.

/* global variables for managing output state */
define variable currLine         as character   no-undo.
define variable currIndent       as integer     no-undo.
define variable nextColumn       as integer     no-undo.

/* parser variables */
define variable parseNum         as integer     no-undo.
define variable currNode         as integer     no-undo.
define variable tempLogical      as logical     no-undo.

/* State variables */
define variable elseBlock        as logical     no-undo initial false.
define variable endBlock         as logical     no-undo initial false.
define variable newBlock         as logical     no-undo initial false.
define variable newState         as logical     no-undo initial true.
define variable trailingSpace    as logical     no-undo initial true.

/* Constant values */
define variable c_False          as integer     no-undo initial 0.
define variable c_True           as integer     no-undo initial 1.
define variable c_NoBlankLine    as integer     no-undo initial 51001.
define variable c_ForceNewline   as integer     no-undo initial 51002.

/* Configurable values - see loadEnv and reformat.ini */
define variable maxLineLen       as integer     no-undo.
define variable wrapLineIndent   as integer     no-undo.
define variable blockIndent      as integer     no-undo.
define variable showLine         as logical     no-undo initial true.
define variable showColumn       as logical     no-undo initial true.
define variable printTokens      as logical     no-undo initial false.
define variable stripComments    as logical     no-undo initial false.

&IF DEFINED(format2PDF) > 0 &THEN
  /* PDF4pro */
  { pdf4pro.i }
&ENDIF

define variable Spdf             as character   no-undo initial "Spdf".
define variable BlockColorRed    as decimal decimals 4 extent 10 no-undo.
define variable BlockColorGreen  as decimal decimals 4 extent 10 no-undo.
define variable BlockColorBlue   as decimal decimals 4 extent 10 no-undo.

  ASSIGN BlockColorRed[ 1]    = 1.0
         BlockColorRed[ 2]    = 0.0
       BlockColorRed[ 3]    = 0.0
       BlockColorRed[ 4]    = 0.5
       BlockColorRed[ 5]    = 0.0
       BlockColorRed[ 6]    = 0.0
       BlockColorRed[ 7]    = 0.1
       BlockColorRed[ 8]    = 0.0
       BlockColorRed[ 9]    = 0.0
       BlockColorRed[10]    = 0.0

       BlockColorGreen[ 1]  = 0.0
       BlockColorGreen[ 2]  = 1.0
       BlockColorGreen[ 3]  = 0.0
       BlockColorGreen[ 4]  = 0.0
       BlockColorGreen[ 5]  = 0.5
       BlockColorGreen[ 6]  = 0.0
       BlockColorGreen[ 7]  = 0.0
       BlockColorGreen[ 8]  = 0.1
       BlockColorGreen[ 9]  = 0.0
       BlockColorGreen[10]  = 0.0

       BlockColorBlue[ 1]   = 0.0
       BlockColorBlue[ 2]   = 0.0
       BlockColorBlue[ 3]   = 1.0
       BlockColorBlue[ 4]   = 0.0
       BlockColorBlue[ 5]   = 0.0
       BlockColorBlue[ 6]   = 0.5
       BlockColorBlue[ 7]   = 0.0
       BlockColorBlue[ 8]   = 0.0
       BlockColorBlue[ 9]   = 0.1
       BlockColorBlue[10]   = 0.0.


/* Proparse */
define variable parser           as handle      no-undo.
{proparse/api/proparse.i parser}

/***************************** Forward Declarations *************************/

function functionState returns logical
   (input piNode as integer) forward.

/* igc */
function triggersState returns logical
   (input piNode as integer) forward.

&IF DEFINED(format2PDF) > 0 &THEN

   function DrawLines returns logical:
      define variable iIndent as integer no-undo.

      /* Draw a line to track block code */
      do iIndent = 1 to currIndent:
        RUN pdf_stroke_color(Spdf,
                             BlockColorRed[iIndent],
                             BlockColorGreen[iIndent],
                             BlockColorBlue[iIndent]).

        run pdf_line_dec(Spdf, 
                        iIndent * 10,
                        pdf_TextY(Spdf) - 2,
                        iIndent * 10,
                        pdf_TextY(Spdf) + 7.0,
                        0.5).
     end.

     return true.

   end function.
&ENDIF

/*********************************  Main Block  *****************************/
   
run proparse/api/proparse.p persistent set parser.

/* Work around initialization bug in Proparse <= 1.2 beta 5 */
output to joanju_empty.p.
output close.
parserParse("joanju_empty.p").
os-delete joanju_empty.p.

run loadEnv.

parseNum = parserParseCreate("scan":U, SEARCH(sourceFileName)).
if parserErrorGetStatus() < 0 then do:
   message parserErrorGetText() view-as alert-box error.
   return.
end.

assign 
   currNode       = parserGetHandle()
   tempLogical    = parserParseGetTop(parseNum, currNode) /* Scanner_head */
  .

&IF DEFINED(format2PDF) > 0 &THEN
  RUN pdf_new(Spdf,targetFileName + ".pdf").

  RUN pdf_set_parameter(Spdf,"Compress","TRUE").
  /* RUN pdf_set_orientation(Spdf,"Landscape"). */
  RUN pdf_set_Font(Spdf,"Courier",8.0).
  RUN pdf_set_LeftMargin(Spdf,10).
  RUN pdf_set_BottomMargin(Spdf,50).

  RUN pdf_new_page(Spdf).
&ENDIF

output to value(targetFileName).

/* do all preprocessing now */
run nodeRefactoring(input currNode).

/* token list goes to the same file for now */
if printTokens then
   run printTokenList(input currNode).

run reformatFile(input currNode).

output close.

&IF DEFINED(format2PDF) > 0 &THEN
  RUN pdf_close(Spdf).
&ENDIF

parserParseDelete(parseNum).

return.


/******************************  functions  ***********************************/
function functionState returns logical
   (input piNode as integer) :
/*------------------------------------------------------------------------------
       Purpose:     Check if we have a function declaration statement or a function 
                    definition block. This is a bit of a hack... it just looks for
                    IN or FORWARDS in the FUNCTION statement
       Parameters:  handle of current node
       Notes:       
------------------------------------------------------------------------------*/
   define variable isBlock    as logical     no-undo initial true.
   define variable iNextNode  as integer     no-undo.
   define variable cNextType  as character   no-undo.
   
   assign
      iNextNode = parserGetHandle()
      cNextType = parserNodeNextSibling(piNode, iNextNode).

   funcLoop:
   do while cNextType <> "":U :
      if cNextType = "PERIOD":U OR cNextType = "LEXCOLON":U then
         leave funcLoop.
      if cNextType = "IN":U OR cNextType = "FORWARDS":U then do:
         assign isBlock = FALSE.
         leave.
      end.
      cNextType = parserNodeNextSibling(iNextNode, iNextNode).
   end.

   parserReleaseHandle(iNextNode).
   
   return isBlock.
end function. /* functionState */

/* igc */
function triggersState returns logical
   (input piNode as integer) :
/*------------------------------------------------------------------------------
       Purpose:     Determine if TRIGGER is beginning of block or end.  This is
                    also a bit of a hack .... the block begins TRIGGERS 
                    statement must end with colon and the END TRIGGERS statement
                    must end with a period.
       Parameters:  handle of current node
       Notes:       
------------------------------------------------------------------------------*/
   define variable isBlock    as logical     no-undo initial false.
   define variable iNextNode  as integer     no-undo.
   define variable cNextType  as character   no-undo.
   
   assign
      iNextNode = parserGetHandle()
      cNextType = parserNodeNextSibling(piNode, iNextNode).

   IF cNextType = "LEXCOLON" THEN
     isBlock = TRUE.

   parserReleaseHandle(iNextNode).
   
   return isBlock.
end function. /* triggersState */

/**************************  Internal Procedures  *****************************/

procedure appendComment:
/*------------------------------------------------------------------------------
       Purpose:     Append a comment to the output
       Parameters:  handle to current Node
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode       as integer no-undo.

   if currLine = "" then do:
      assign currLine = parserGetNodeText(piNode).
      run putLine(input piNode).
   end.
   else
      assign currLine = currLine + "~n":U + parserGetNodeText(piNode) + "~n":U.
end procedure. /* appendComment */



procedure appendText:
/*------------------------------------------------------------------------------
       Purpose:     Append text to the current line.
       Parameters:  handle to current Node
                    currentText to display
                    number of spaces to follow
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode       as integer no-undo.
   define input parameter currNodeText as character no-undo.
   define input parameter numSpaces    as integer no-undo.

   assign nextColumn = nextColumn + length(currNodeText).
   if nextColumn > maxLineLen OR parserAttrGetI(piNode, c_ForceNewline) = c_True then do:
      assign
         currLine    = currLine + "~n" + fill(" ":U, wrapLineIndent)
         nextColumn  = wrapLineIndent + length(currNodeText).
   end. /* wrap line */
   else if currLine > "" AND trailingSpace then
      assign currLine = currLine + fill(" ":U, numSpaces).

   assign
      currLine       = currLine + currNodeText
      newState       = false
      trailingSpace  = true
      .
end procedure. /* appendText */



procedure assignStatement:
/*------------------------------------------------------------------------------
       Purpose:     watch for where to put line breaks into an assign statement.
       Parameters:  handle to current Node
       Notes:       We put a line break in front of every identifier or system handle
                    that is not preceded by an operator, leftparen, etc. This is 
                    pretty rough right now - there are probably examples of 
                    expression code that this would not handle well.
------------------------------------------------------------------------------*/
   define input parameter piNode       as integer no-undo.

   define variable followsOperator  as logical     no-undo initial false.
   define variable iTempNode        as integer     no-undo.
   define variable cTempType        as character   no-undo.
   
   iTempNode = parserGetHandle().

   assign cTempType = parserNodeNextSibling(piNode, iTempNode).
   assignLoop:
   do while cTempType <> "":
      if cTempType = "PERIOD":U then
         leave assignLoop.
      if lookup(cTempType, "WS,NEWLINE,COMMENT":U) > 0 then do:
         assign cTempType = parserNodeNextSibling(iTempNode, iTempNode).
         next assignLoop.
      end.

      if lookup(cTempType,
      "NAMEDOT,OBJCOLON,COMMA,OR,AND,NOT,MATCHES,BEGINS,CONTAINS,EQUAL,EQ,GTORLT,NE,RIGHTANGLE,GTHAN,GTOREQUAL,GE,LEFTANGLE,LTHAN,LTOREQUAL,LE,PLUS,MINUS,STAR,SLASH,MODULO,MINUS,PLUS,LEFTPAREN,LEFTBRACE,LEFTCURLY":U
      ) > 0 then do:
         assign
            cTempType         = parserNodeNextSibling(iTempNode, iTempNode)
            followsOperator   = true
         .
         next assignLoop.
      end.

      if lookup(cTempType,
      "ID,AAMEMORY,ACTIVEWINDOW,CLIPBOARD,CODEBASELOCATOR,COLORTABLE,COMPILER,COMSELF,CURRENTWINDOW,DEBUGGER,DEFAULTWINDOW,ERRORSTATUS,FILEINFORMATION,FOCUS,FONTTABLE,LASTEVENT,MOUSE,PROFILER,RCODEINFORMATION,SELF,SESSION,SOURCEPROCEDURE,TARGETPROCEDURE,TEXTCURSOR,THISPROCEDURE,WEBCONTEXT":U
      ) > 0 AND NOT followsOperator then
         parserAttrSet(iTempNode, c_ForceNewline, c_True).

      assign
         cTempType         = parserNodeNextSibling(iTempNode, iTempNode)
         followsOperator   = false
      .
   end.
   
   parserReleaseHandle(iTempNode).
end procedure. /* assignStatement */



procedure gatherComment:
/*------------------------------------------------------------------------------
       Purpose:     Gather all tokens within a comment into one proper COMMENT token.
       Parameters:  handle to current node
       Notes:       
------------------------------------------------------------------------------*/
   define input-output parameter piNode as integer no-undo.
   
   define variable commentLevel  as integer     no-undo initial 1.
   define variable commentText   as character   no-undo.
   define variable iPlaceNode    as integer     no-undo.
   define variable iNextNode     as integer     no-undo.
   define variable cNodeType     as character   no-undo.
   define variable cNextType     as character   no-undo.

   assign
      iPlaceNode  = parserGetHandle()
      iNextNode   = parserGetHandle().

   parserCopyHandle(piNode, iPlaceNode).
   
   assign 
      commentText = parserGetNodeText(piNode)
      cNodeType   = parserGetNodeType(piNode).
   
   comment-loop:
   do while cNodeType <> "":
      assign cNodeType = parserNodeNextSibling(piNode, piNode).
      if commentLevel = 0 then
         leave comment-loop.
      if cNodeType = "NEWLINE":U then do:
         /* We strip whitespace that follows a newline, because later
         * on we will want to manually deal with the indenting, except...
         * We keep one space in front of a STAR, and three
         * in front of anything else.
         */
         assign
         commentText = commentText + "~n":U
         cNextType = parserNodeNextSibling(piNode, iNextNode)
         .
         if cNextType = "WS":U then do:
            assign 
               cNodeType   = parserNodeNextSibling(piNode, piNode)
               cNextType   = parserNodeNextSibling(piNode, iNextNode)
               commentText = if cNextType = "STAR":U then commentText + " ":U
                             else commentText + "   ":U.
         end.
         next comment-loop.
      end.
      assign commentText = commentText + parserGetNodeText(piNode).

      if cNodeType = "COMMENTSTART":U then do:
         assign commentLevel = commentLevel + 1.
         next comment-loop.
      end.

      if cNodeType = "COMMENTEND":U
         /* Begin workaround a bug in Proparse's "scanner", version <= 1.2 beta 5... */
         OR INDEX(parserGetNodeText(piNode), "*/") > 0 then do:
         /* ...end of bug workaround */
         assign commentLevel = commentLevel - 1.
         next comment-loop.
      end.
   end. /* comment-loop */

   if cNodeType = "":U then
      return "Unmatched comment start found" + substring(commentText, 60).

   /* modify node text and type */
   parserSetNodeText(iPlaceNode, commentText).
   parserSetNodeType(iPlaceNode, "COMMENT":U).
   parserSetNodeNextSibling(iPlaceNode, piNode).
   
   parserReleaseHandle(iPlaceNode).
   parserReleaseHandle(iNextNode).
end procedure. /* gatherComment */



procedure gatherPreprocessReference:
/*------------------------------------------------------------------------------
       Purpose:     Gather all tokens that compose one reference to a preprocessor tag.
       Parameters:  handle to current node
       Notes:       collects the pieces and makes a PREPROCESSTOKEN
------------------------------------------------------------------------------*/
   define input-output parameter piNode as integer no-undo.
   
   define variable preProcText   as character   no-undo.
   define variable iPlaceNode    as integer     no-undo.
   define variable iCurrNode     as integer     no-undo.
   define variable cNodeType     as character   no-undo.

   assign
      iPlaceNode  = parserGetHandle()
      iCurrNode   = parserGetHandle().

   parserCopyHandle(piNode, iPlaceNode).
   parserCopyHandle(piNode, iCurrNode).
   
   assign 
      preProcText = parserGetNodeText(iCurrNode)
      cNodeType   = parserGetNodeType(iCurrNode).
   
   preProcess-loop:
   do while cNodeType <> "":

      assign 
         cNodeType   = parserNodeNextSibling(iCurrNode, iCurrNode)
         preProcText = preProcText + parserGetNodeText(iCurrNode)
      .

      if cNodeType = "RIGHTCURLY":U then do:
         /* advance pointer beyond the 'rightcurly' */
         cNodeType   = parserNodeNextSibling(iCurrNode, iCurrNode).
         leave preProcess-loop.
      end.
   end. /* preProcess-loop */

   /* modify node text and type */
   parserSetNodeText(iPlaceNode, preProcText).
   parserSetNodeType(iPlaceNode, "PREPROCESSTOKEN":U).
   parserSetNodeNextSibling(iPlaceNode, iCurrNode).
   parserCopyHandle(iCurrNode, piNode).
   
   parserReleaseHandle(iPlaceNode).
   parserReleaseHandle(iCurrNode).
end procedure. /* gatherPreprocessReference */



procedure gatherString:
/*------------------------------------------------------------------------------
       Purpose:     Gather all tokens within a quoted string into one proper QSTRING 
                    token.
       Parameters:  handle to current node
       Notes:       
------------------------------------------------------------------------------*/
   define input-output parameter piNode as integer no-undo.
   
   define variable currString    as character   no-undo.
   define variable quoteType     as character   no-undo.
   define variable quoteText     as character   no-undo.
   define variable iPlaceNode    as integer     no-undo.
   define variable iNextNode     as integer     no-undo.
   define variable cNodeType     as character   no-undo.
   define variable cNextType     as character   no-undo.

   assign
      iPlaceNode  = parserGetHandle()
      iNextNode   = parserGetHandle().

   assign 
      cNodeType   = parserGetNodeType(piNode)
      quoteType   = cNodeType
      quoteText   = parserGetNodeText(piNode)
      currString  = quoteText
      .

   parserCopyHandle(piNode, iPlaceNode).

   quote-loop:
   do while cNodeType <> "":
      assign cNodeType = parserNodeNextSibling(piNode, piNode).

      /* Watch for quoted quote */
      if cNodeType = quoteType then do:
         assign cNextType = parserNodeNextSibling(piNode, iNextNode).
         if cNextType = quoteType then do:
            assign currString = currString + quoteText + quoteText.
            parserNodeNextSibling(piNode, piNode).
            assign cNodeType = parserNodeNextSibling(piNode, piNode).
            next quote-loop.
         end.
         else do:
            assign
               currString = currString + quoteText
               cNodeType = parserNodeNextSibling(piNode, piNode)
            .
            leave quote-loop.
         end.
      end.

     /* Watch for escape sequence
      * All we have to do is print it, and then fall through to
      * adding the next token.
      */
      if cNodeType = "TILDE":U OR cNodeType = "BACKSLASH":U then do:
         assign
            currString  = currString + parserGetNodeText(piNode)
            cNodeType   = parserNodeNextSibling(piNode, PiNode)
         .
      end.
   
      assign
         currString  =  if cNodeType = "NEWLINE":U then currString + "~n"
                        else currString + parserGetNodeText(piNode).

   end. /* quote-loop */

   if cNodeType = "":U then
      return "Unmatched quote found".

   /* String attributes */
   if cNodeType = "OBJCOLON" then do:
      assign
         currString  = currString + ":":U
         cNodeType   = parserNodeNextSibling(piNode, piNode)
         currString  = currString + parserGetNodeText(piNode)
         cNodeType   = parserNodeNextSibling(piNode, piNode)
      .
   end.

   parserSetNodeText(iPlaceNode, currString).
   parserSetNodeType(iPlaceNode, "QSTRING":U).
   if cNodeType <> "":U then
      parserSetNodeNextSibling(iPlaceNode, piNode).

   parserReleaseHandle(iPlaceNode).
   parserReleaseHandle(iNextNode).
   return.
end procedure. /* gatherString */


procedure loadEnv:
/*------------------------------------------------------------------------------
       Purpose:     Load configuration settings from reformat.ini
       Parameters:  <none>
       Notes:       
------------------------------------------------------------------------------*/
   define variable envFile    as character no-undo.
   define variable envPath    as character no-undo.
   define variable tempChar   as character no-undo.
   define variable tempInt    as integer no-undo.

   assign envFile = search("reformat/reformat.ini":U).
   if envFile = ? then
      assign envFile = search("reformat/reformat.default.ini":U).
   if envFile = ? then do:
      message "Could not find reformat.ini.":T view-as alert-box error.
      return.
   end.

   assign
      envFile = REPLACE(envFile, "~\":U, "/":U)
      tempInt = R-INDEX(envFile, "/":U)
      envPath = SUBSTRING(envFile, 1, tempInt - 1)
      envFile = SUBSTRING(envFile, tempInt + 1)
   .

   load envFile
      dir envPath
      base-KEY "INI":U
   .

   use envFile.
  
   get-key-value
      section "lines"
      key "max-line-len"
      value tempChar.
   assign maxLineLen = integer(tempChar).

   get-key-value
      section "lines"
      key "wrap-line-indent"
      value tempChar.
   assign wrapLineIndent = integer(tempChar).

   get-key-value
      section "lines"
      key "block-indent"
      value tempChar.
   assign blockIndent = integer(tempChar).

   get-key-value
      section "comments"
      key "strip-comments"
      value tempChar.
   assign stripComments = (tempChar = "yes" OR tempChar = "true").

   get-key-value
      section "printing"
      key "show-line"
      value tempChar.
   assign showLine = (tempChar = "yes" OR tempChar = "true").

   get-key-value
      section "printing"
      key "show-column"
      value tempChar.
   assign showColumn = (tempChar = "yes" OR tempChar = "true").

   get-key-value
      section "printing"
      key "print-tokens"
      value tempChar.
   assign printTokens = (tempChar = "yes" OR tempChar = "true").
end procedure. /* loadEnv */



procedure markupElse:
/*------------------------------------------------------------------------------
       Purpose:     Markups for ELSE node.
       Parameters:  <none>
       Notes:       We find the previous statement end, because we don't want a 
                    blank line between an if..then block and an ELSE block.
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.
   
   define variable iTempNode  as integer     no-undo.
   define variable cTempType  as character   no-undo.
   
   assign
      iTempNode = parserGetHandle()
      cTempType = parserNodePrevSibling(piNode, iTempNode).

   elseLoop:
   do while cTempType <> "":
      if cTempType = "LEXCOLON":U OR cTempType = "PERIOD":U then do:
         parserAttrSet(iTempNode, c_NoBlankLine, c_True).
         leave elseLoop.
      end.
      assign cTempType = parserNodePrevSibling(iTempNode, iTempNode).
   end.
   
   parserReleaseHandle(iTempNode).
end procedure. /* markupElse */



procedure putLine:
/*------------------------------------------------------------------------------
       Purpose:     Write the current line out.
       Parameters:  handle to current Node
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.
   
   define variable iLines   as integer no-undo.
   define variable iLine    as integer no-undo.

   if newBlock and not elseBlock then do:
      &IF DEFINED(format2PDF) > 0 &THEN
        run pdf_skip(Spdf).
        DrawLines().
      &ENDIF
      
      put unformatted skip(1).
   end.

   if endBlock then
      assign currIndent = currIndent - 1.

   /* igc - stop it from going negative --- this mainly happened due to issue
            with TRIGGERS statement but better to fix it for 'everything' */
   if currIndent < 0 then
     assign currIndent = 0.

   /* if this is a broken line, then we have to indent at the line breaks */
   assign currLine = replace(currLine, "~n", "~n" + fill(" ":U, blockIndent * currIndent)).

   &IF DEFINED(format2PDF) > 0 &THEN
     if index(currLine,"~n") > 0 then do:
        iLines = NUM-ENTRIES(currline,"~n").

        do iLine = 1 to iLines:
          run pdf_skip(Spdf).
          DrawLines().

          run pdf_text(Spdf,  fill(" ":U, blockIndent * currIndent)
                            + entry(iLine,currLine,"~n")).
        end.

        run pdf_skip(Spdf).
        DrawLines().

     end.
     else do:
       run pdf_skip(Spdf).
       DrawLines().

       run pdf_text(Spdf,  fill(" ":U, blockIndent * currIndent)
                         + currLine).
     end.
   &ENDIF
   
   put unformatted skip space(blockIndent * currIndent) currLine.

   if newBlock then
      assign currIndent = currIndent + 1.
   if endBlock and parserAttrGetI(piNode, c_NoBlankLine) = c_False then do:
      &IF DEFINED(format2PDF) > 0 &THEN
        run pdf_skip(Spdf).
        DrawLines().
      &ENDIF
      
      put unformatted skip(1).
   end.

   assign
      currLine    = "":U
      nextColumn  = blockIndent * currIndent + 1
      newBlock    = false
      endBlock    = false
      elseBlock   = false
   .
end procedure. /* putLine */

PROCEDURE printTokenList:
/*------------------------------------------------------------------------------
       Purpose:     Prints entire list of tokens, one node per line
       Parameters:  handle of root node
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.

   define variable iCurrNode as integer   no-undo.
   define variable cNodeType as character no-undo.
   
   iCurrNode = parserGetHandle().
   parserCopyHandle(piNode, iCurrNode).
   cNodeType = parserGetNodeType(iCurrNode).
   
   DO WHILE cNodeType <> "":
      &IF DEFINED(format2PDF) > 0 &THEN
         run pdf_text(Spdf,  parserGetNodeType(iCurrNode)
                           + "    ":U
                           + parserGetNodeText(iCurrNode) ).

         if showLine or showColumn then
           run pdf_text(Spdf, substitute("    &1", string(parserGetNodeLine(iCurrNode))) ).

         if showColumn then 
           run pdf_text(Spdf, substitute(":&1", string(parserGetNodeColumn(iCurrNode))) ).

         run pdf_skip(Spdf).
         DrawLines().
      &ENDIF

      PUT UNFORMATTED
         parserGetNodeType(iCurrNode)
         "    ":U
         parserGetNodeText(iCurrNode)

         if showLine or showColumn then substitute("    &1", string(parserGetNodeLine(iCurrNode)))
         else ""

         if showColumn then substitute(":&1", string(parserGetNodeColumn(iCurrNode)))
         else ""

         skip
      .
      
      cNodeType = parserNodeNextSibling(iCurrNode, iCurrNode).
   END.

   parserReleaseHandle(iCurrNode).
END PROCEDURE. /* printline */




procedure preprocessDirective:
/*------------------------------------------------------------------------------
       Purpose:     Write preprocess directives to the end of the line, unless we 
                    hit an escaped newline.
       Parameters:  handle of current node
       Notes:       We don't split these lines - that would make the code uncompilable.
------------------------------------------------------------------------------*/
   define input-output parameter piNode as integer no-undo.

   define variable cNodeType as character no-undo.
   
   cNodeType = parserGetNodeType(piNode).
   
   put skip.
   preproloop:
   do while cNodeType <> "":
      if cNodeType = "TILDE":U OR cNodeType = "BACKSLASH":U then do:

         &IF DEFINED(format2PDF) > 0 &THEN
            run pdf_text(Spdf,parserGetNodeText(piNode)).
         &ENDIF
         
         put unformatted parserGetNodeText(piNode). 
         assign cNodeType = parserNodeNextSibling(piNode, piNode).

         if cNodeType = "NEWLINE":U then do:
           &IF DEFINED(format2PDF) > 0 &THEN
              run pdf_skip(Spdf).
              DrawLines().
           &ENDIF
           
           put skip. 
         end.

         else do:
           &IF DEFINED(format2PDF) > 0 &THEN
              run pdf_text(Spdf, parserGetNodeText(piNode)).
           &ENDIF
           
            put unformatted parserGetNodeText(piNode).
         end.

         assign cNodeType = parserNodeNextSibling(piNode, piNode).
         next preproloop.
      end.

      if cNodeType = "NEWLINE":U then do:
         &IF DEFINED(format2PDF) > 0 &THEN
            run pdf_skip(Spdf).
            DrawLines().
         &ENDIF

         put skip.
         leave preproloop.
      end.

      &IF DEFINED(format2PDF) > 0 &THEN
         run pdf_text(Spdf,parserGetNodeText(piNode)).
      &ENDIF
      
      put unformatted parserGetNodeText(piNode).

      assign cNodeType = parserNodeNextSibling(piNode, piNode).
   end. /* preproloop */
end procedure. /* preprocessDirective */



procedure preproEndif:
/*------------------------------------------------------------------------------
       Purpose:     Write &ELSE or &ENDIF on its own line.
       Parameters:  handle of current node
       Notes:       We don't do any indenting for these beasts.
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.

   define variable currLineLen as integer no-undo.
   if currLine <> "" then do:
      assign
         currLineLen = LENGTH(currLine)
         currLine    = currLine + "~n" + parserGetNodeText(piNode) + "~n" + fill(" ":U, currLineLen)
      .
   end.
   else do:
      &IF DEFINED(format2PDF) > 0 &THEN
         run pdf_skip(Spdf).
         DrawLines().
         run pdf_text(Spdf,parserGetNodeText(piNode)).
         run pdf_skip(Spdf).
         DrawLines().
      &ENDIF
      
      put unformatted skip parserGetNodeText(piNode) skip.
   end.
end procedure. /* preproEndif */



procedure preproIf:
/*------------------------------------------------------------------------------
       Purpose:     Write preprocess &if to the &then.
       Parameters:  handle of current node
       Notes:       We don't do any indenting or line truncation for these beasts.
------------------------------------------------------------------------------*/
   define input-output parameter piNode as integer no-undo.

   define variable cNodeType     as character no-undo.
   define variable ifText        as character no-undo.
   define variable currLineLen   as integer no-undo.
   
   define variable iLines        as integer no-undo.
   define variable iLine         as integer no-undo.

   cNodeType = parserGetNodeType(piNode).
   
   preproloop:
   do while cNodeType <> "":
      if cNodeType = "TILDE":U OR cNodeType = "BACKSLASH":U then do:
         assign
            ifText      = ifText + parserGetNodeText(piNode)
            cNodeType   = parserNodeNextSibling(piNode, piNode)
         .
         ifText = if cNodeType = "NEWLINE":U then ifText + "~n"
                  else ifText + parserGetNodeText(piNode).
         assign
            cNodeType = parserNodeNextSibling(piNode, piNode).
         next preproloop.
      end.

      assign 
         ifText = ifText + parserGetNodeText(piNode).
      if cNodeType = "AMPTHEN":U then
         leave preproloop.
      else
         assign cNodeType = parserNodeNextSibling(piNode, piNode).
         
   end. /* preproloop */
   
   if currLine <> "" then do:
      assign
         currLineLen = LENGTH(currLine)
         currLine    = currLine + "~n" + ifText + "~n" + fill(" ":U, currLineLen)
      .
   end.
   else do:

      &IF DEFINED(format2PDF) > 0 &THEN
         if index(ifText,"~n") > 0 then do:
           iLines = num-entries(ifText,"~n").

           do iLine = 1 to iLines:
             run pdf_skip(Spdf).
             DrawLines().

             run pdf_text(Spdf, ENTRY(iLine, ifText, "~n")).
           end.
         end.

         else do:
           run pdf_skip(Spdf).
           DrawLines().
           run pdf_text(Spdf,ifText).
         end.

         run pdf_skip(Spdf).
         DrawLines().
      &ENDIF
      
      put unformatted skip ifText skip.
   end.
end procedure. /* preproIf */


procedure lookForPreprocessorsAndArgs:
/*------------------------------------------------------------------------------
       Purpose:     pass for lexical cleanup of preprocessor references
       Parameters:  handle of root node
       Notes:       We should really build up temptable of pre-processor names.
                    This would allow us to distinguish between the named arguments
                    to include files and SCOPED/GLOBAL defined names.  This temp-
                    table should include the effective 'scope' of all elements.
                    That is, named arguments are automatically scoped to the include
                    file.  &SCOPED and &GLOBAL defines have their respective scopes,
                    subject to &UNDEFINE statements.
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.
   
   define variable cNodeType as character no-undo.
   define variable iCurrNode as integer   no-undo.
   
   assign
      iCurrNode = parserGetHandle()
      cNodeType = parserNodeNextSibling(piNode, iCurrNode) /* first real token */
   .
   
   curlyLoop:
   do while cNodeType <> "":

      case cNodeType:

         when "CURLYAMP":U then do:
            run gatherPreprocessReference(input-output iCurrNode).
            cNodeType = parserGetNodeType(iCurrNode).
            next curlyLoop.
         end.

         when "CURLYNUMBER":U then do:
            run gatherPreprocessReference(input-output iCurrNode).
            cNodeType = parserGetNodeType(iCurrNode).
            next curlyLoop.
         end.

      end case.

      assign 
         cNodeType = parserNodeNextSibling(iCurrNode, iCurrNode).

      if parserErrorGetStatus() < 0 then do:
         message parserErrorGetText() view-as alert-box error.
         leave curlyLoop.
      end.

   end. /* curlyLoop */
   
   parserReleaseHandle(iCurrNode).
end procedure.


procedure lookForCommentsAndStrings:
/*------------------------------------------------------------------------------
       Purpose:     pass for lexical cleanup to do some things that Proparse's 
                    scanner does not attempt
       Parameters:  handle of root node
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.

   define variable cNodeType as character no-undo.
   define variable iCurrNode as integer   no-undo.
   
   assign
      iCurrNode = parserGetHandle()
      cNodeType = parserNodeNextSibling(piNode, iCurrNode) /* first real token */
   .
   
   commentQuoteLoop:
   do while cNodeType <> "":

      case cNodeType:

         when "COMMENTSTART":U then do:
            run gatherComment(input-output iCurrNode).
            if return-value <> "":u then do:
               message return-value view-as alert-box error.
               return.
            end.
            cNodeType = parserGetNodeType(iCurrNode).
            next commentQuoteLoop.
         end.

         when "DOUBLEQUOTE":U or 
         when "SINGLEQUOTE":U then do:
            run gatherString(input-output iCurrNode).
            if return-value <> "":u then do:
               message return-value view-as alert-box error.
               return.
            end.
            cNodeType = parserGetNodeType(iCurrNode).
            next commentQuoteLoop.
         end.

      end case.

      assign 
         cNodeType = parserNodeNextSibling(iCurrNode, iCurrNode).

      if parserErrorGetStatus() < 0 then do:
         message parserErrorGetText() view-as alert-box error.
         leave commentQuoteLoop.
      end.

   end. /* commentQuoteLoop */
   
   parserReleaseHandle(iCurrNode).
end procedure.


procedure lookForAssignsAndElses:
/*------------------------------------------------------------------------------
       Purpose:     pass for lexical cleanup to look for some things that
                    are a little bit tricky, like where to break assign statements.
       Parameters:  handle of root node
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.

   define variable cNodeType as character no-undo.
   define variable iCurrNode as integer   no-undo.
   
   assign
      iCurrNode = parserGetHandle()
      cNodeType = parserNodeNextSibling(piNode, iCurrNode) /* first real token */
   .
   
   assignElseLoop:
   do while cNodeType <> "":

      case cNodeType:

         when "assign":U then do:
            run assignStatement(input iCurrNode).
         end.

         when "ELSE":U then do:
            run markupElse(input iCurrNode).
         end.

      end case.

      assign cNodeType = parserNodeNextSibling(iCurrNode, iCurrNode).

      if parserErrorGetStatus() < 0 then do:
         message 
            parserErrorGetText() 
         view-as alert-box error.
         leave assignElseLoop.
      end.

   end. /* assignElseLoop */
   
   parserReleaseHandle(iCurrNode).
end procedure.

procedure nodeRefactoring:
/*------------------------------------------------------------------------------
       Purpose:     to gather fragmented primitive tokens into more meaningful 
                    chunks
       Parameters:  handle of root node
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.
   
   run lookForPreprocessorsAndArgs(input piNode).
/* run printTokenList(input piNode).               */
   run lookForCommentsAndStrings(input piNode).
   run lookForAssignsAndElses(input piNode).
end procedure.

procedure reformatFile:
/*------------------------------------------------------------------------------
       Purpose:     this is where the heavy-litfing is done
       Parameters:  handle of root node
       Notes:       
------------------------------------------------------------------------------*/
   define input parameter piNode as integer no-undo.

   define variable cNodeType as character no-undo.
   define variable iCurrNode as integer   no-undo.
   
   assign
      iCurrNode = parserGetHandle()
      cNodeType = parserNodeNextSibling(piNode, iCurrNode) /* first real token */
   .
   
   reformatLoop:
   do while cNodeType <> "":

      case cNodeType:

         when "WS":U or
         when "NEWLINE":U then do:
            /* do nothing */
         end.

         when "PREPROCESSTOKEN":U then do:
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 0).
         end.

         when "COMMENT":U then do:
            if NOT stripComments then run appendComment(input iCurrNode).
         end.

         when "AMPANALYZESUSPEND":U OR 
         when "AMPANALYZERESUME":U  OR 
         when "AMPGLOBALDEFINE":U   OR 
         when "AMPUNDEFINE":U       OR 
         when "AMPMESSAGE":U        OR 
         when "AMPSCOPEDDEFINE":U
            then run preprocessDirective(input-output iCurrNode).

         when "AMPIF":U OR 
         when "AMPELSEIF":U then 
            run preproIf(input-output iCurrNode).

         when "AMPELSE":U OR 
         when "AMPENDIF":U then 
            run preproEndif(input iCurrNode).

         when "OBJCOLON":U then do:
            assign cNodeType = parserNodeNextSibling(iCurrNode, iCurrNode).
            run appendText(input iCurrNode, input ":":U + parserGetNodeText(iCurrNode), input 0).
         end.

         when "NAMEDOT":U then do:
            assign cNodeType = parserNodeNextSibling(iCurrNode, iCurrNode).
            run appendText(input iCurrNode, input ".":U + parserGetNodeText(iCurrNode), input 0).
         end.

         when "PERIOD":U OR 
         when "LEXCOLON":U then do:
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 0).
            run putLine(input iCurrNode).
            assign newState = TRUE.
         end.

         when "THEN":U then do:
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 1).
            assign newState = TRUE.
         end.

         when "ELSE":U then do:
            assign elseBlock = TRUE.
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 1).
            assign newState = TRUE.
         end.

         when "PROCEDURE":U OR 
         when "DO":U        OR 
         when "REPEAT":U    OR 
         when "FOR":U       
         then do:

            /* DO block might also be in an ON trigger */
            if newState OR cNodeType = "DO":U then
            assign newBlock = TRUE.
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 1).
         end.

         when "FUNCTION":U then do:
            if newState then do:
               /* Check if this is a function block or function statement */
               newBlock = functionState(input iCurrNode).
            end.
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 1).
         end.

         /* igc - added to handle TRIGGERS on widgets */
         when "TRIGGERS":U then do:
            newBlock = triggersState(input iCurrNode).
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 1).
         end.
        
         when "END":U then do:
            if newState then
            assign endBlock = TRUE.
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 1).
         end.

         when "COMMA":U OR 
         when "RIGHTPAREN":U then
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 0).

         when "LEFTPAREN":U then do:
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 0).
            assign trailingSpace = FALSE.
         end.

         /* igc - Added to put out a new line after } - otherwise caused 
                  indentation issue when CREATE widget TRIGGERS phrase was 
                  encountered */
         when "RIGHTCURLY":U then do:
           run appendText(input iCurrNode, input parserGetNodeText(iCurrNode) + "~n", input 0).
           run putLine(input iCurrNode).
           assign newState = TRUE.
         end.

         when "LEFTCURLY":U then do:
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 0).
            assign trailingSpace = FALSE.
         end.

         otherwise
            run appendText(input iCurrNode, input parserGetNodeText(iCurrNode), input 1).

      end case.

      assign cNodeType = parserNodeNextSibling(iCurrNode, iCurrNode).

      if parserErrorGetStatus() < 0 then do:
         message parserErrorGetText() view-as alert-box error.
         leave reformatLoop.
      end.

   end. /* reformatLoop */

   run putLine(input iCurrNode).

end procedure.

