/* GuiBomUnit.groovy
July 2008 by John Green

Copyright (C) 2008 Joanju Software (www.joanju.com). All rights reserved.
This file is made available under the terms of the Eclipse Public License v1.0.
 */
import org.prorefactor.treeparser.ParseUnit
import groovy.xml.MarkupBuilder
import org.prorefactor.treeparser.Symbol
import org.prorefactor.widgettypes.Frame
import com.joanju.proparse.IntegerIndex
import com.joanju.proparse.NodeTypes
import org.prorefactor.core.JPNode
import org.prorefactor.core.TreeUtils

class GuiBomUnit {

	GuiBomUnit(ParseUnit parseUnit, ScriptSupport support) {
		this.parseUnit = parseUnit
		this.support = support
		nodeArray = TreeUtils.nodeArray(parseUnit.topNode)
	}

	JPNode [] nodeArray
	ParseUnit parseUnit
	ScriptSupport support

	def elementIndexes = new IntegerIndex<Object>()
	def frames = new HashSet<Frame>()
	def referencingNodes = new HashMap<Symbol, HashSet<JPNode>>()
	def statements = new LinkedHashSet<JPNode>()
	def widgets = new HashSet<Symbol>();

	def writer = new StringWriter()
	def builder = new MarkupBuilder(writer)




	void build() {
		builder.'guibom'('file': support.relativePath(parseUnit.getFile())) {
			// Build list of fileId to fileName for reference
			builder.'sourcefiles'() {
				listSourceFiles()
			}
			// Build lists of frames and their widgets.
			builder.'frames'() {
				listFrames()
			}
			// Next find nodes which reference those frames and widgets.
			findReferencingNodes()
			builder.'widgets'() {
				listWidgets()
			}
			builder.'statements'() {
				writeStatements()
			}
		}
	}


	void findReferencingNodes() {
		for (JPNode node in nodeArray) {
			Symbol s = node.symbol
			if (s==null)
				continue
			if (widgets.contains(s) || frames.contains(s)) {
				elementIndexes.add(node)
				statements.add(node.getStatement())
				referencingNodes.get(s).add(node)
			}
		}
	}


	String getText() {return writer.toString()}


	void listFrames() {
		for (Frame frame in parseUnit.getRootScope().getAllSymbolsDeep(Frame)) {
			def index = elementIndexes.add(frame)
			frames.add(frame)
			referencingNodes.put(frame, new HashSet<JPNode>())
			builder.'frame'('id':index, 'name':frame.name) {
				for (Symbol symbol in frame.getAllFieldsAndWidgets()) {
					widgets.add(symbol)
					referencingNodes.put(symbol, new HashSet<JPNode>())
					def index2 = elementIndexes.add(symbol)
					def type = NodeTypes.getFullText(symbol.getProgressType());
					def enabled = frame.enabledFields.contains(symbol)
					def attrs = ['id':index2, 'type':type, 'name':symbol.getName()]
					if (enabled)
						attrs += ['enabled':enabled]
					builder.'widget'(attrs)
				}
				for (JPNode node in frame.statementList) {
					def attrs = [
							'id': elementIndexes.add(node),
							'state': NodeTypes.getFullText(node.type),
							'line': node.line,
							'column': node.column,
							'fileid': node.fileIndex
							]
					if (node.state2>0)
						attrs += ['state2': NodeTypes.getFullText(node.state2)]
					builder.'statement'(attrs)
				}
			}
		}
	}


	void listSourceFiles() {
		def filenames = parseUnit.getFileIndex()
		def end = filenames.length - 1
		for (int i in 0 .. end) {
			builder.'sourcefile'(
					'fileid': i,
					'filename': support.relativePath(new File(filenames[i]))
					)
		}
	}


	void listWidgets() {
		for (symbol in widgets) {
			def index = elementIndexes.getIndex(symbol)
			def type = NodeTypes.getFullText(symbol.getProgressType());
			def attrs = ['id':index, 'type':type, 'name':symbol.getName()]
			builder.'widget'(attrs) {
				for (node in referencingNodes.get(symbol)) {
					// Many nodes are 'synthetic' - inserted into the tree parser
					// for tree structure, and don't come from tokens from source.
					// So, to get a sensible file/line/column here, we
					// use firstNaturalChild() which returns itself if natural,
					// or if synthetic, then the first natural child.
					JPNode natural = node.firstNaturalChild()
					builder.'referencingnode'(
							'id': elementIndexes.getIndex(node),
							)
				}
			}
		}
	}


	void writeStatements() {
		for (node in statements) {
			writeSyntaxTree(node, false)
		}
	}


	void writeSyntaxTree(JPNode node, boolean showSiblings) {
		if (node==null)
			return
		def attrs = [:]
		if (elementIndexes.hasValue(node))
			attrs += ['id': elementIndexes.getIndex(node)]
		String type
		if (NodeTypes.isKeywordType(node.type))
			type = NodeTypes.getFullText(node.type)
		else
			type = NodeTypes.getTypeName(node.type)
		attrs += ['type': type]
		Symbol symbol = node.symbol
		if (symbol!=null) {
			if (frames.contains(symbol))
				attrs += ['frameid': elementIndexes.getIndex(symbol)]
			else if (widgets.contains(symbol)) {
				attrs += ['widgetid': elementIndexes.getIndex(symbol)]
			}
		}
		attrs += ['fileid':node.fileIndex, 'line':node.line, 'column':node.column]
		if (node.isNatural())
			attrs += ['text': node.text]
		builder.'node'(attrs) {
			writeSyntaxTree(node.firstChild(), true)
		}
		if (showSiblings)
			writeSyntaxTree(node.nextSibling(), true)
	}


}
