/* joanju/filerefs.p
 * April 2007 by John Green
 *
 * Builds temp-tables of file references (for RUN and for include).
 * What you do with those temp-tables is entirely up to you.
 *
 * Look for CHANGE-THIS in the comments in the code for portions that
 * must be customized for your project.
 *
 * This program writes various strings to the screen for no other reasons
 * than status display and for debugging.
 * 
 * Copyright (C) 2007 Joanju Software.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at http://www.eclipse.org/legal/epl-v10.html.
 */

/* --- Begin temp-tables which contain program results. --- */

/* List of .p program names which failed to parse. */
def temp-table parseFailedTable no-undo
    field filename as character.
def buffer parseFailed for parseFailedTable.

/* List of names of files which are used as include files. */
def temp-table includeFileTable no-undo
    field filename as character
    index idx1 is unique filename.
def buffer includeFile for includeFileTable.

/* RUN VALUE statements which might need more analysis. */
def temp-table runValueTable no-undo
    field expression as character
    field sourceFile as character
    field sourceLine as integer
    index idx1 is unique sourceFile sourceLine.
def buffer runValue for runValueTable.

/* List of names of files which are RUN targets. */
def temp-table externalProcedureTable no-undo
    field filename as character
    index idx1 is unique filename.
def buffer externalProcedure for externalProcedureTable.

/* --- End of temp-tables which contain program results. --- */


DEFINE VARIABLE parser AS HANDLE NO-UNDO.
RUN proparse/api/proparse.p PERSISTENT SET parser.
{proparse/api/proparse.i parser}

{joanju/allfilestable.i}
def buffer possibleProgramFile for allFilesTable.

def temp-table internalProceduresTable no-undo
    field procname as character
    index idx1 procname.
def buffer internalProcedure for internalProceduresTable.


form with frame f2 side-labels.

function expressionText returns character (input nodeHandle as integer) forward.


if parserGetVersion() < "3.1" then do:
  message "Proparse version 3.1 (minimum) is required." view-as alert-box.
  return.
end.


pause 0 before-hide.
display "Fetching possible program file names..." format "x(75)" with frame f0.
/* CHANGE-THIS with the appropriate directories for your project. */
run joanju/programfiles.p (input "com", input-output table allFilesTable append).
run joanju/programfiles.p (input "loc", input-output table allFilesTable append).
run joanju/programfiles.p (input "mapics.ht", input-output table allFilesTable append).
run joanju/programfiles.p (input "varnet", input-output table allFilesTable append).
hide frame f0.

display "Press Ctrl-break to stop." with frame helpFrame.

for each possibleProgramFile:

  display possibleProgramFile.filename format "x(75)" no-label with frame f1.
  parserParse(possibleProgramFile.filename).

  if (parserErrorGetStatus() <> 0) then do:
    /* Parse failed. Keep track of it if it was a .p. */
    if (possibleProgramFile.filename matches "*~~.p") then do:
      display possibleProgramFile.filename format "x(65)" label "Parse failed" with frame f2.
      create parseFailed.
      parseFailed.filename = possibleProgramFile.filename.
    end.
    next.
  end.

  def var topHandle as integer no-undo.
  topHandle = parserGetHandle().
  parserNodeTop(topHandle).

  run addIncludeFiles(topHandle).
  run buildInternalProcedureList(topHandle).
  run processRunStatements(topHandle).
  
end.

/* CHANGE-THIS: You have to do something with the data from the temp-tables here.
 * This program only builds the temp-tables, it doesn't print them or anything.
 */
/* ---- example ----
output to "includerefs.txt".
for each includeFile:
  if not includeFile.filename matches "*~~.i" then
     put unformatted
       includeFile.filename
       ", "
       includeFile.filename + ".i"
       skip.
end.
output close.
output to "programrefs.txt".
for each externalProcedure:
  if not externalProcedure.filename matches "*~~.p" then
     put unformatted
       externalProcedure.filename
       ", "
       externalProcedure.filename + ".p"
       skip.
end.
output close.
---- end example ---- */
 
return.


procedure addIncludeFiles:
  def input parameter topHandle as integer no-undo.
  def var filenameList as character no-undo.
  filenameList = parserAttrGet(topHandle, "filename-list").
  def var entryNum as integer no-undo.
  /* The first entry is the compile unit filename itself. We don't need it. */
  do entryNum = 2 to num-entries(filenameList, chr(10)):
    def var includeFilename as character no-undo.
    includeFilename = entry(entryNum, filenameList, chr(10)).
    if (not can-find(includeFile where includeFile.filename eq includeFilename)) then do:
      create includeFile.
      includeFile.filename = includeFilename.
      display includeFilename format "x(65)" label "Include"
          with frame f2.
    end.
  end.
end procedure.


procedure buildInternalProcedureList:
  def input parameter topHandle as integer no-undo.
  def var procHandle as integer no-undo.
  def var idHandle as integer no-undo.
  procHandle = parserGetHandle().
  idHandle = parserGetHandle().
  def var counter as integer no-undo.
  def var numResults as integer no-undo.
  numResults = parserQueryCreate(topHandle, "procedures", "PROCEDURE").
  for each internalProcedure: delete internalProcedure. end.
  do counter = 1 to numResults:
    parserQueryGetResult("procedures", counter, procHandle).
    /** Check that it's a PROCEDURE statement, and not some other PROCEDURE node. */
    if (parserAttrGet(procHandle, "statehead") ne "t") then
        next.
    parserNodeFirstChild(procHandle, idHandle).
    create internalProcedure.
    internalProcedure.procname = parserGetNodeText(idHandle).
    display internalProcedure.procname format "x(65)" label "Internal"
        with frame f2.
  end.
end procedure.


function expressionText returns character (input topHandle as integer):
  /* Create an "unfiltered query" to get a flattened list of nodes
   * that make up this expression, and then build a string from those.
   */
  def var resultHandle as integer no-undo.
  resultHandle = parserGetHandle().
  def var counter as integer no-undo.
  def var numResults as integer no-undo.
  def var retString as character no-undo.
  numResults = parserQueryCreate(topHandle, "flat_expression", "").
  do counter = 1 to numResults:
    parserQueryGetResult("flat_expression", counter, resultHandle).
    def var nodeText as character no-undo.
    nodeText = parserGetNodeText(resultHandle).
    if (nodeText ne "") then
        retString = retString + " " + nodeText.
  end.
  return trim(retString).
end function.


procedure processRunStatements:
  def input parameter topHandle as integer no-undo.
  def var runHandle as integer no-undo.
  def var idHandle as integer no-undo.
  def var valueHandle as integer no-undo.
  runHandle = parserGetHandle().
  idHandle = parserGetHandle().
  valueHandle = parserGetHandle().
  def var counter as integer no-undo.
  def var numResults as integer no-undo.
  numResults = parserQueryCreate(topHandle, "runs", "RUN").
  runs-loop: do counter = 1 to numResults:
    parserQueryGetResult("runs", counter, runHandle).
    /* Check that it's a RUN statement, and not some other RUN node. */
    if (parserAttrGet(runHandle, "statehead") ne "t") then
        next runs-loop.
    /* We are only interested in vanilla RUN statements,
     * not RUN STORED-PROCEDURE or RUN SUPER.
     */
    if (parserAttrGet(runHandle, "state2") ne "") then
       next runs-loop.
    def var nodeType as character no-undo. 
    nodeType = parserNodeFirstChild(runHandle, idHandle).
    def var runText as character no-undo.
    if (nodeType eq "VALUE") then do:
      /* We want expression from #(VALUE LEFTPAREN expression ... */
      parserNodeFirstChild(idHandle, valueHandle).
      parserNodeNextSibling(valueHandle, valueHandle).
      /* Watch for RUN VALUE("hard/coded/filename"). Why people do this is beyond me. */
      if (parserGetNodeType(valueHandle) eq "QSTRING") then do:
        runText = parserGetNodeText(valueHandle).
      end. else do:
        run processRunValue(runHandle, valueHandle).
        next runs-loop.
      end.
    end. else do:
      runText = parserGetNodeText(idHandle).
    end.
    
    /* The filename in RUN filename might be quoted, so watch for quotes and strip them.
     * Note: doesn't deal with string attributes - not needed for this particular project.
     * CHANGE-THIS: Add support for string attributes if you need it.
     */
    def var firstChar as character no-undo.
    firstChar = substring(runText,1,1).
    if (firstChar eq '"' or firstChar eq "'") then
        runText = substring(runText, 2, length(runText) - 2).
    
    if (can-find(internalProcedure where internalProcedure.procname eq runText)) then
        next runs-loop.
    if (can-find(externalProcedure where externalProcedure.filename eq runText)) then
        next runs-loop.
    create externalProcedure.
    externalProcedure.filename = runText.
    display externalProcedure.filename format "x(65)" label "External"
        with frame f2.
  end. /* runs-loop */
end procedure.


procedure processRunValue:
  def input parameter runHandle as integer no-undo.
  def input parameter valueHandle as integer no-undo.
  if (can-find(runValue
      where runValue.sourceFile = parserGetNodeFilename(runHandle)
      and runValue.sourceLine = parserGetNodeLine(runHandle)
      )) then
      return.
  create runValue.
  runValue.expression = expressionText(valueHandle).
  runValue.sourceFile = parserGetNodeFilename(runHandle).
  runValue.sourceLine = parserGetNodeLine(runHandle).
  display
      runValue.expression format "x(60)" label "Run val"
      runValue.sourceFile format "x(60)" label "Run val src"
      runValue.sourceLine label "Run val line"
      with frame f2
      .
end procedure.

