/* Code Documenter - generates HTML documentation of programs.  
autodox2.p   2001 by Judy Hoffman Green, Joanju Limited

Parses and documents significant features of each program listed,
displays the results out to an edit window, and generates 
the html documentation for these programs.

This procedure must be run persistently.

In character mode, the editor "ed1" has a different background 
color when it is disabled from when it is enabled. This may cause
a flicker effect when running a small test set of programs as the 
editor is enabled and the background color changes, but no information is lost. 


Methods List
============

main
      No parameters.
      Run this once everything is set.
getNumErrors
      OUTPUT INTEGER
      Get the number of errors after calling "main".
setAppName  
      INPUT CHARACTER
      Sets the name of your application.
setOutputDir
      INPUT CHARACTER
      Sets the name of the output directory. Leave blank for current 
      working dir.
setFileList
      INPUT CHARACTER
      Sets the name of a file containing a list of program names
      (.w, .p, etc). One program name per line. Lines can
      be commented out by inserting lines containing nothing
      but open comment (/*) and closing comment (*/).
      The program names in this file must be either relative to your working
      directory or have fully-qualified paths.
setProcCommentLoc
      INPUT LOGICAL
      Sets whether to look Before or After the PROCEDURE statement for a 
      comment.
setFuncCommentLoc
      INPUT LOGICAL
      Sets whether to look Before or After the FUNCTION statement for a comment.
setParamCommentLoc
      INPUT LOGICAL
      Sets whether to look Before or After the DEFINE PARAMETER statement 
      for a comment.
*/


DEFINE VARIABLE parserHandle AS HANDLE NO-UNDO.
RUN proparse/api/proparse.p PERSISTENT SET parserHandle.
{proparse/api/proparse.i parserHandle}


DEFINE VARIABLE fileList        AS CHARACTER NO-UNDO.
DEFINE VARIABLE outputDir       AS CHARACTER NO-UNDO.
DEFINE VARIABLE appName         AS CHARACTER NO-UNDO.
DEFINE VARIABLE procCommentLoc  AS LOGICAL   NO-UNDO.
DEFINE VARIABLE funcCommentLoc  AS LOGICAL   NO-UNDO.
DEFINE VARIABLE paramCommentLoc AS LOGICAL   NO-UNDO.

DEFINE VARIABLE logoFile        AS CHARACTER NO-UNDO.
DEFINE VARIABLE prevDir         AS CHARACTER NO-UNDO.
DEFINE VARIABLE currDir         AS CHARACTER NO-UNDO.
DEFINE VARIABLE currFile        AS CHARACTER NO-UNDO.
DEFINE VARIABLE errFile         AS CHARACTER NO-UNDO.
DEFINE VARIABLE indFile         AS CHARACTER NO-UNDO.
DEFINE VARIABLE tocFile         AS CHARACTER NO-UNDO.
DEFINE VARIABLE sumFile         AS CHARACTER NO-UNDO.
DEFINE VARIABLE detFile         AS CHARACTER NO-UNDO.
DEFINE VARIABLE detOutFile      AS CHARACTER NO-UNDO.
DEFINE VARIABLE detHeader       AS CHARACTER NO-UNDO.
DEFINE VARIABLE detFooter       AS CHARACTER NO-UNDO.
 
DEFINE VARIABLE errorMess       AS CHARACTER NO-UNDO.
DEFINE VARIABLE fileName        AS CHARACTER NO-UNDO.
DEFINE VARIABLE commentLevel    AS INTEGER   NO-UNDO.
DEFINE VARIABLE counter         AS INTEGER   NO-UNDO.
DEFINE VARIABLE numErrors       AS INTEGER   NO-UNDO.
DEFINE VARIABLE firstNode       AS INTEGER   NO-UNDO.
DEFINE VARIABLE i1              AS INTEGER   NO-UNDO.
DEFINE VARIABLE i2              AS INTEGER   NO-UNDO.
DEFINE VARIABLE ed1             AS WIDGET-HANDLE NO-UNDO.
DEFINE VARIABLE edHeight        AS INTEGER   NO-UNDO.
DEFINE VARIABLE lComment        AS CHARACTER NO-UNDO.
DEFINE VARIABLE sComment        AS CHARACTER NO-UNDO.
DEFINE VARIABLE funcText        AS CHARACTER NO-UNDO.
DEFINE VARIABLE start           AS INTEGER   NO-UNDO.

DEFINE VARIABLE numResults      AS INTEGER   NO-UNDO.
DEFINE VARIABLE found           AS LOGICAL   NO-UNDO.
DEFINE VARIABLE dummy           AS LOGICAL   NO-UNDO.
DEFINE VARIABLE grandChild      AS INTEGER   NO-UNDO.

DEFINE STREAM errStream.
DEFINE STREAM Ssum.
DEFINE STREAM Sdet.
DEFINE STREAM Stoc.
DEFINE STREAM Sind.

DEFINE TEMP-TABLE Tproc NO-UNDO
  FIELD procName    AS CHARACTER 
  FIELD procComment AS CHARACTER 
  INDEX pn IS UNIQUE PRIMARY procName.

DEFINE TEMP-TABLE Tfunc NO-UNDO
  FIELD funcName    AS CHARACTER 
  FIELD funcComment AS CHARACTER 
  FIELD funcText    AS CHARACTER 
  INDEX pn IS UNIQUE PRIMARY funcName.

DEFINE TEMP-TABLE Tparam NO-UNDO
  FIELD procName     AS CHARACTER 
  FIELD paramComment AS CHARACTER 
  FIELD paramType AS CHARACTER 
  FIELD paramName AS CHARACTER 
  FIELD dataType  AS CHARACTER 
  INDEX prm IS UNIQUE PRIMARY procName paramType paramName.


FORM WITH FRAME f1 VIEW-AS DIALOG-BOX SIZE 122 BY 27 NO-LABELS THREE-D.


ON "CLOSE" OF THIS-PROCEDURE DO:
  DELETE PROCEDURE THIS-PROCEDURE.
END.


RETURN.  /* Run me persistent! */





PROCEDURE main: 


  /* Prepare the output HTML files: */
  
  ASSIGN 
    errFile = SESSION:TEMP-DIRECTORY + "joanju_autodox_err.txt"
    indFile = "index.html"
    tocFile = "toc.html"
    sumFile = "summary.html"
    detFile = "detail".   /* later assigned a num + ".html" ("detail27.html") */
  
  OUTPUT STREAM Stoc TO VALUE(outputDir + tocFile).
  OUTPUT STREAM Ssum TO VALUE(outputDir + sumFile).

  RUN writeIndexFile.
  RUN writeTocFileHeader.
  RUN writeSumFileHeader.

  RUN prepareDetHtml.
  

  
  IF SESSION:DISPLAY-TYPE = "TTY" THEN 
    ASSIGN edheight = 22.
  ELSE  
    ASSIGN edheight = 25.
  
  CREATE EDITOR ed1
    ASSIGN
      READ-ONLY = TRUE
      FRAME = FRAME f1:HANDLE
      SCROLLBAR-VERTICAL = TRUE
      SCROLLBAR-HORIZONTAL = TRUE
      LARGE = TRUE
      WIDTH = 120
      HEIGHT = edheight.
  DISPLAY WITH FRAME f1.
  
  ed1:INSERT-STRING("Press Esc to stop.~n").
  


  /* Copy logo to output directory. */
  ASSIGN logoFile = SEARCH("autodox/autodoxsmall.jpg").

  IF logoFile = ? THEN DO: 
    ASSIGN 
      errorMess = "Could not find AutoDox logo file autodox/autodoxsmall.jpg". 
    ed1:INSERT-STRING("~n" + errorMess + "~n").
    RUN errorLogger ("~n" + errorMess).
  END.

  OS-COPY VALUE(logoFile) VALUE(outputDir + "autodoxsmall.jpg").
  IF OS-ERROR <> 0 THEN DO:
    ASSIGN errorMess = "Error during copy of logo: OS error #" 
                       + STRING(OS-ERROR,"999").
    ed1:INSERT-STRING("~n" + errorMess + "~n").
    RUN errorLogger ("~n" + errorMess).
  END.

 
  
  INPUT FROM VALUE( SEARCH(fileList) ).
  
  
  ASSIGN counter = 0.
  
  
  
  
  file-loop:
  REPEAT ON ERROR UNDO, LEAVE:

    PROCESS EVENTS.
  
    ASSIGN counter = counter + 1.
    IMPORT UNFORMATTED filename.
    IF filename = "" THEN
       NEXT file-loop.
  
    IF filename = "/*" THEN
      ASSIGN commentlevel = commentlevel + 1.
  
    IF commentlevel = 0 AND filename <> "*/" THEN 
    DO:
  
      /* If the filename is very long, and the whole line gets too long for 
       * the EDITOR, then the entire editor "shifts" right, and it looks 
       * ugly - you can't see the line number anymore.
       * Using WORD-WRAP doesn't resolve this nicely, because then the editor
       * doesn't display any text until it has an entire line to write out.
       */
      ed1:INSERT-STRING(
          "~n"
          + STRING(counter) + " "
          + IF LENGTH(filename,"COLUMN") > 50 THEN
               "..." 
               + SUBSTRING(filename, LENGTH(filename,"COLUMN") - 45,45,"COLUMN")
            ELSE
               filename
          + "  parse"
          ).
  
      FILE-INFO:FILE-NAME = filename.
      IF FILE-INFO:FULL-PATHNAME = ? THEN DO:
        ASSIGN errorMess = SUBSTITUTE("file &1 not found",filename).
        ed1:INSERT-STRING("~n" + errorMess + "~n").
        RUN errorLogger (errorMess).
        NEXT file-loop.
      END.
      IF parserParse(filename) = FALSE THEN DO:
        ASSIGN errorMess = parserErrorGetText().
        ed1:INSERT-STRING("~n" + errorMess + "~n").
        RUN errorLogger (filename + "~n" + errorMess).
        NEXT file-loop.
      END.
  



      /* Begin the documentation. */

      ed1:INSERT-STRING(" document"). 
  
      ASSIGN  
        firstNode  = parserGetHandle()
        grandChild = parserGetHandle().
  
      parserNodeTop(firstNode).  /* gets us the "Program_root" node */
  
  
      /* Empty the temp-table records from the previous filename iteration. */
      FOR EACH Tproc:  DELETE Tproc. END. 
      FOR EACH Tfunc:  DELETE Tfunc. END. 
      FOR EACH Tparam: DELETE Tparam. END. 
  

      /* Break apart the path and filename for tidier display. */
      ASSIGN 
        start    = MAX(R-INDEX(fileName,"/"), R-INDEX(fileName,"~\")) 
        currDir  = SUBSTRING(fileName,1,start)
        currFile = SUBSTRING(fileName,start + 1).




      /* There are five main steps to document a program: 
       * 
       * Step 1. Figure out the program's description (in a comment) if 
       *         we can find one. 
       * 
       * For further information, see readme.html section "Finding the 
       * First Comment...".
       */
      RUN getProcComment.



  
      /* Step 2. Go through the internal procedures to pick up their 
       *         names and parameters. 
       */
      RUN getProcedures. 


  

      /* Step 3. Go through the whole program to pick up its' own parameters. 
       *  
       * Run getParams again, this time to populate TT records for the all 
       * of the parameters of the program itself, not its' procedures' 
       * parameters. 
       */
      RUN getParams (INPUT firstNode, 
                     INPUT "",      /* No procedure name */
                     INPUT FALSE).  /* Get attribute instead of setting it. */
  
  

  
      /* Step 4. Go through the program to pick up the functions. */
      RUN getFunctions. 

  

  
      /* Check for errors here, just before we do step 5, the output. */
      IF parserErrorGetStatus() <> 0 THEN DO:
        ASSIGN errorMess = parserErrorGetText().
        ed1:INSERT-STRING("~n" + errorMess + "~n").
        RUN errorLogger (filename + "~n" + errorMess).
        NEXT file-loop.
      END.
  
    


      /* Step 5. Output the resulting documentation for this program. 
       *         We're writing out to three separate streams here, and 
       *         generating multiple .html files. 
       */
  
      ASSIGN detOutFile = detFile + TRIM(STRING(counter)) + ".html".
 

      /* Write the program name information to the TOC and summary pages. */
      RUN checkTocDir. 
      RUN writeTocFileLine.

      RUN checkSumDir. 
      RUN writeSumFileLine.


      ASSIGN prevDir = currDir. 
     
      /* Write the detail page for this program. */ 
      RUN writeDetFile.




      /* We're finished documenting this program. */
      ed1:INSERT-STRING(" ok").
  
    END. /* if commentlevel = 0... */
  
  
  
    IF filename = "*/" THEN
      ASSIGN commentlevel = commentlevel - 1.
  
  END. /* file-loop */
  


  INPUT CLOSE.
  

  RUN writeTocFileFooter.
  RUN writeSumFileFooter.


  /* Close the two streams still open. */
  OUTPUT STREAM Stoc CLOSE.
  OUTPUT STREAM Ssum CLOSE.
  
  
  APPLY "CLOSE" TO parserHandle.
  
  
  IF numErrors > 0 THEN
     ed1:INSERT-STRING("~n~nError logfile will be displayed next.~n").
  ELSE
     ed1:INSERT-STRING("~n~nThere were no errors.~n").
  
  ed1:INSERT-STRING("~nPress Esc to close.~n").
  
  ENABLE ALL WITH FRAME f1.
  DO ON ERROR UNDO, LEAVE ON END-KEY UNDO, LEAVE:
    WAIT-FOR WINDOW-CLOSE OF FRAME f1.
  END.


  IF numErrors > 0 THEN DO:
    PUT STREAM errStream UNFORMATTED
      SKIP(1)
      "Number of errors: "
      numErrors
      SKIP
      "Press Esc to close.".
    OUTPUT STREAM errStream CLOSE.
    RUN proparse/utilities/resultswindow.p (errFile).
  END.

  RETURN.

END PROCEDURE. /* main */





PROCEDURE errorLogger:
  DEFINE INPUT PARAMETER theText AS CHARACTER NO-UNDO.
  IF numErrors = 0 THEN DO:
    OUTPUT STREAM errStream TO VALUE(errFile).
    PUT STREAM errStream UNFORMATTED
      errFile
      SKIP
      "AutoDox error log".
  END.
  ASSIGN numErrors = numErrors + 1.
  PUT STREAM errStream UNFORMATTED SKIP(1) theText.
END PROCEDURE.





PROCEDURE getProcedures: 

  /* Go through the internal procedures to pick up their names 
   * and parameters. 
   */

  DEFINE VARIABLE comment    AS CHARACTER NO-UNDO.
  DEFINE VARIABLE queryChild AS INTEGER   NO-UNDO.
  DEFINE VARIABLE idChild    AS INTEGER   NO-UNDO.

  ASSIGN 
    queryChild = parserGetHandle()
    idChild    = parserGetHandle()
    numResults = parserQueryCreate(firstNode, "query_proc", "PROCEDURE").
 
  
  /* Go through all the PROCEDURE statements. */
  DO i1 = 1 TO numResults:
      
    parserQueryGetResult("query_proc", i1, queryChild).

    /* If there's no child or the child isn't type "ID", it's not 
     * an internal PROCEDURE. 
     */
    IF parserNodeFirstChild(queryChild,idChild) <> "ID" 
    THEN NEXT.

    /* Just take the procedure's first definition. */
    IF CAN-FIND(FIRST Tproc WHERE Tproc.procName = parserGetNodeText(idChild))
    THEN NEXT.


    /* Get a comment for this internal procedure, if possible.
     * For further information, see the readme.html section 
     * "Looking for Comments". 
     */
    IF procCommentLoc = TRUE THEN    /* Before */
      RUN getCommentBefore(INPUT queryChild, OUTPUT comment). 
    ELSE     /* After */
      RUN getBlockComment(queryChild, OUTPUT comment).


    /* Replace characters significant to HTML in the comment text. */
    RUN replaceChars (INPUT-OUTPUT comment). 

    /* Gather the above information into the temp-table for later output. */
    CREATE Tproc. 
    ASSIGN Tproc.procName    = parserGetNodeText(idChild)
           Tproc.procComment = comment. 

    /* Get all of the params in this proc node only, populating the TT */
    RUN getParams (INPUT queryChild, 
                   INPUT Tproc.procName, 
                   INPUT TRUE). 

  END. /* Do i1 = 1 ... each of the PROCEDURE statements. */

END PROCEDURE. /* getProcedures */





PROCEDURE getFunctions: 

  /* Go through the program to pick up the functions. */

  DEFINE VARIABLE comment    AS CHARACTER NO-UNDO.
  DEFINE VARIABLE queryChild AS INTEGER   NO-UNDO.
  DEFINE VARIABLE idChild    AS INTEGER   NO-UNDO.

  ASSIGN 
    queryChild = parserGetHandle()
    idChild    = parserGetHandle()     
    numResults = parserQueryCreate(firstNode,"query_funcs","FUNCTION").

      
  /* Go through all the FUNCTION statements. */
  DO i1 = 1 TO numResults:
  
    parserQueryGetResult("query_funcs", i1, queryChild).

    /* If there's no child or the child isn't type "ID", we don't want it */
    IF parserNodeFirstChild(queryChild,idChild) <> "ID" 
    THEN NEXT.

    /* Get a comment for this function, if possible.
     * For further information, see the readme.html section 
     * "Looking for Comments". 
     */
    IF funcCommentLoc = TRUE THEN    /* Before */
      RUN getCommentBefore(INPUT queryChild, OUTPUT comment). 
    ELSE     /* After */
      RUN getBlockComment(queryChild, OUTPUT comment).

    /* Replace characters significant to HTML in the comment text. */
    RUN replaceChars (INPUT-OUTPUT comment). 

    /* Get the text of the function call. */
    ASSIGN funcText = "".
    RUN getStmtText (INPUT queryChild, INPUT-OUTPUT funcText, OUTPUT dummy). 

    /* See if we've already got this function from a forward
     * declaration. We use the text from the second time 'round
     * (the definition), but if we have a comment from before and none
     * from now, then we preserve the comment.
     */
    FIND Tfunc WHERE Tfunc.funcName = parserGetNodeText(idChild) NO-ERROR.
    IF AVAILABLE Tfunc THEN DO:
      IF comment = "" THEN ASSIGN comment = Tfunc.funcComment.
      DELETE Tfunc.
    END.

    /* Gather the above information into the temp-table for later output. */
    CREATE Tfunc. 
    ASSIGN Tfunc.funcName    = parserGetNodeText(idChild)
           Tfunc.funcComment = comment
           Tfunc.funcText    = funcText. 

  END. /* Do i1 = 1 ... each of the FUNCTION statements. */

END PROCEDURE. /* getFunctions */





PROCEDURE getParams: 

/* Find all of the parameters in the main procedure or internal procedure, 
 * build TT recs.
 * Pass in parameter pProcName, which is either blank or a procedure name.
 * Pass in parameter pSetProcAttr for whether we're setting this attribute 
 * or checking it. When setting the attribute, this marks the nodes as 
 * being inside a procedure. If we're checking the attribute, this is for 
 * when we are looking for only non-procedure nodes, so we discard all 
 * nodes that were previously marked.
 */

  DEFINE INPUT PARAMETER pStartNode   AS INTEGER   NO-UNDO. 
  DEFINE INPUT PARAMETER pProcName    AS CHARACTER NO-UNDO. 
  DEFINE INPUT PARAMETER pSetProcAttr AS LOGICAL   NO-UNDO. 
 
  DEFINE VARIABLE queryChild   AS INTEGER   NO-UNDO.
  DEFINE VARIABLE child        AS INTEGER   NO-UNDO.
  DEFINE VARIABLE child2       AS INTEGER   NO-UNDO.
  DEFINE VARIABLE numResults2  AS INTEGER   NO-UNDO.
  DEFINE VARIABLE numChildren  AS INTEGER   NO-UNDO.
  DEFINE VARIABLE haveNode     AS LOGICAL   NO-UNDO.
  DEFINE VARIABLE haveParam    AS LOGICAL   NO-UNDO.
 
  DEFINE VARIABLE nodeType     AS CHARACTER NO-UNDO.
  DEFINE VARIABLE paramType    AS CHARACTER NO-UNDO.
  DEFINE VARIABLE paramName    AS CHARACTER NO-UNDO.
  DEFINE VARIABLE dataType     AS CHARACTER NO-UNDO.
  DEFINE VARIABLE paramComment AS CHARACTER NO-UNDO.


  ASSIGN 
    queryChild = parserGetHandle()
    child      = parserGetHandle()
    child2     = parserGetHandle().


  /* Find all of the DEFINE nodes of this PROCEDURE. */
  numResults2 = parserQueryCreate(pStartNode, "query_define", "DEFINE"). 


  /* Go through all the DEFINE statements in the given node. */
  DO i2 = 1 TO numResults2:
      
    parserQueryGetResult("query_define", i2, queryChild).

    /* If pSetProcAttr is TRUE, mark this DEFINE node as being in a PROCEDURE 
     * even if it isn't a PARAMETER node. This way, in a later call to this
     * internal procedure with pSetProcAttr = FALSE, we can discard all DEFINE 
     * nodes without checking if they're DEFINE PARAMETER nodes first.
     */
    IF pSetProcAttr THEN 
      parserAttrSet(queryChild,50001,2).
    ELSE  
      IF parserAttrGetI(queryChild,50001) = 2 THEN NEXT.
       
       
    ASSIGN  
      haveNode     = parserNodeFirstChild(queryChild,child) <> "" 
      numChildren  = 0
      haveParam    = FALSE
      nodeType     = ""
      paramType    = ""
      paramName    = ""
      dataType     = ""
      paramComment = "".



    /* Go through all children of the DEFINE node, 
     * e.g. "INPUT" "PARAMETER" "Pcust" "AS (CHARACTER)" etc.,
     * gathering data about this DEFINE statement or discarding it if 
     * it isn't a PARAMETER statement.
     */
    DO WHILE haveNode:  

      ASSIGN 
        numChildren = numChildren + 1
        nodeType    = parserGetNodeType(child).

      IF nodeType = "PARAMETER" THEN 
        ASSIGN haveParam = TRUE.

      /* Check now to get out of here as fast as possible if not a parameter.*/
      IF numChildren >= 2 AND haveParam = FALSE THEN LEAVE. 


      /* e.g. it's an INPUT parameter. */
      IF CAN-DO("INPUT,OUTPUT,RETURN",nodeType) THEN
        ASSIGN paramType = nodeType.

      /* Tidy up for INPUT-OUTPUT parameters. */
      ELSE
      IF nodeType = "INPUTOUTPUT" THEN
        ASSIGN paramType = "INPUT-OUTPUT".

      ELSE  /* e.g. "Pcust" */
      IF nodeType = "ID" THEN 
        ASSIGN paramName = parserGetNodeText(child).

      ELSE 
      IF nodeType = "AS" THEN DO:  
        ASSIGN dataType = nodeType. 
        IF parserNodeFirstChild(child,child2)  <> ""  
        OR parserNodeNextSibling(child,child2) <> "" THEN DO:
          ASSIGN dataType = dataType + " " + parserGetNodeText(child2).
        END.
      END. 

      ELSE 
      IF nodeType = "LIKE" THEN DO: 
        ASSIGN dataType = nodeType. 
        IF parserNodeFirstChild(child,child2) <> "" THEN /* Shd always exist */
          ASSIGN dataType = dataType + " " + parserGetNodeText(child2).
      END. 

      ELSE        /* e.g.: "DEFINE PARAMETER BUFFER <ID> FOR <RECORD_NAME> */
      IF nodeType = "BUFFER" THEN DO: 
        ASSIGN paramType = "BUFFER".

        /* next sib is the ID token, which we skip, as handled in ID section. */
        IF parserNodeNextSibling(child,child2) <> "" THEN DO:  
          IF parserNodeNextSibling(child2,child2) <> "" THEN DO: /* the "FOR" */
            ASSIGN dataType = parserGetNodeText(child2).
            IF parserNodeNextSibling(child2,child2) <> "" THEN /*"RECORD_NAME"*/
              ASSIGN dataType = dataType + " " + parserGetNodeText(child2).
          END.
        END.
      END. 


      /* We're appending the details of this type of define statement to the 
       * paramType variable, and the paramName and dataType variables will 
       * be blank.
       */
      ELSE    /* e.g. "DEFINE INPUT PARAMETER TABLE FOR <RECORD_NAME> */
      IF nodeType = "TABLE" THEN DO: 
        ASSIGN paramType = paramType + " " + nodeType.

        /* next sibling is the FOR token, which we want. */
        IF parserNodeNextSibling(child,child2) <> "" THEN DO:  /* the "FOR" */
          ASSIGN paramType = paramType + " " + parserGetNodeText(child2).
          IF parserNodeNextSibling(child2,child2) <> "" THEN /* "RECORD_NAME" */
            ASSIGN paramType = paramType + " " + parserGetNodeText(child2).
        END.
      END. 


      /* We're appending the details of this type of define statement to the 
       * paramType variable, the paramName will contain the ID, and the 
       * dataType variable will be blank.
       */
      ELSE         /* e.g. "DEFINE INPUT PARAMETER TABLE-HANDLE FOR ID */
      IF nodeType = "TABLEHANDLE" THEN DO: 
        ASSIGN paramType = paramType + " TABLE-HANDLE".  /* append to "INPUT" */

        /* The next sibling might be the optional FOR token, which we want, or 
         * it might be the ID, which we don't want as it will be handled above. 
         */
        IF parserNodeNextSibling(child,child2) = "FOR" THEN
          ASSIGN paramType = paramType + " " + parserGetNodeText(child2).
      END. 


/* **** John Green (TODO) hiddenGetAfter is no longer available ****
      /* If your parameter comments are After the end of the parameter 
       * statement, go find them now while you are working with the 
       * correct node. 
       * For further information, see readme.html section "Looking for Comments"
       */
      ELSE      
      IF  nodeType = "PERIOD" 
      AND paramCommentLoc = FALSE THEN   /* After */
        RUN getCommentAfter (INPUT child, INPUT TRUE, OUTPUT paramComment). 
**** John Green (TODO) hiddenGetAfter is no longer available **** */


      ASSIGN haveNode = (parserNodeNextSibling(child,child) <> "").

    END.  /* do while haveNode - each of the children of the DEFINE node. */


    /* If we have a DEFINE ... PARAMETER statement, then create a TT record. */
    IF haveParam = TRUE THEN DO: 

      /* If your parameter comments are Before the parameter statement, 
       * go find them now while you are working with the correct node. 
       * For further information, see readme.html section "Looking for Comments"
       */
      IF paramCommentLoc = TRUE THEN   /* Before */
        RUN getCommentBefore (INPUT queryChild, OUTPUT paramComment). 

      /* Replace characters significant to HTML in the comment text. */
      RUN replaceChars (INPUT-OUTPUT paramComment). 

      CREATE Tparam. 
      ASSIGN Tparam.procName     = pProcName   
             Tparam.paramComment = paramComment
             Tparam.paramType    = paramType
             Tparam.paramName    = paramName 
             Tparam.dataType     = dataType.

    END. /* if haveParam */

  END. /* DO i2... each of the DEFINE statements in the PROCEDURE. */

  /* Unload unneeded query. */
  parserQueryClear("query_define").

END PROCEDURE. /* getParams */





PROCEDURE getProcComment: 

/* Figure out the program's description (in a comment) if we can find one. 
 * For further information, see the readme.html section "Finding the First 
 * Comment...".
 */

  DEFINE VARIABLE comChild AS INTEGER NO-UNDO.

  ASSIGN  
    comChild = parserGetHandle()
    lComment = ""
    sComment = "". 

  
  /* Find the first real node in the program. */
  RUN getRealNode (INPUT firstNode, OUTPUT comChild, OUTPUT dummy).
  

  /* dummy = FALSE means there were no real nodes in the file being processed, 
   * perhaps because the file contains only comments. This should be quite rare.
   */
  IF dummy = TRUE THEN DO: 

    /* For a .p, this program looks for the first comment in the program. */
    IF SUBSTRING(filename,LENGTH(filename) - 1) = "~.p" THEN DO: 

      ASSIGN found = parserHiddenGetBefore(comChild).
      DO WHILE found:  
        IF parserHiddenGetType() = "COMMENT" THEN
          ASSIGN lComment = parserHiddenGetText().
        ASSIGN found = parserHiddenGetPrevious().
      END. 

      /* Replace characters significant to HTML in the comment text. */
      RUN replaceChars (INPUT-OUTPUT lComment). 

      /* Take the first line, minus the first / *, for the short comment. 
       * If the comment also ended on this line, remove the trailing * and /. 
       */
      IF lComment <> "" THEN DO:
        ASSIGN sComment = SUBSTRING(ENTRY(1,lComment,"~n"),3). 
        IF SUBSTRING(sComment,LENGTH(sComment) - 1) = "*/" THEN
          ASSIGN sComment = SUBSTRING(sComment,1,LENGTH(sComment) - 2). 
      END.

    END. /* if substring... .p filenames */


    /* For a .w or other extension, this program looks for the text 
     * "Description: " in a comment before the first real node in the .w. 
     */
    ELSE DO:

      ASSIGN found = parserHiddenGetBefore(comChild).
      DO WHILE found:  
        IF parserHiddenGetType() = "COMMENT" THEN DO:
          ASSIGN 
            lComment = parserHiddenGetText()
            start    = INDEX(lComment, "Description: ").
          IF start <> 0 THEN DO: 
            ASSIGN sComment = SUBSTRING(lComment,start + 13,60). 
            /* If there's multiple lines in sComment, only take the first line. */
            IF INDEX(sComment,"~n") <> 0 THEN 
              ASSIGN sComment = SUBSTRING(sComment,1,INDEX(sComment,"~n") - 1).
            LEAVE. 
          END.
        ELSE ASSIGN lComment = "".
        END.
        ASSIGN found = parserHiddenGetPrevious().
      END. 

      /* Replace characters significant to HTML in the comment text. */
      RUN replaceChars (INPUT-OUTPUT lComment). 

    END. /* else do:  .w filenames */

  END. /* IF dummy = TRUE */
 
END PROCEDURE. /* getProcComment */





PROCEDURE getCommentBefore: 

/* Procedure to find a hidden comment token before the given node, 
 * and return its' text. 
 */

  DEFINE INPUT  PARAMETER pNode    AS INTEGER   NO-UNDO. /* node to look bef */
  DEFINE OUTPUT PARAMETER pComment AS CHARACTER NO-UNDO. /* text of comment */

  IF parserHiddenGetBefore(pNode) = TRUE THEN DO: 
    IF parserHiddenGetType() = "COMMENT" THEN
      ASSIGN pComment = parserHiddenGetText().

    ELSE DO WHILE parserHiddenGetPrevious() = TRUE: 
      IF parserHiddenGetType() = "COMMENT" THEN DO: 
        ASSIGN pComment = parserHiddenGetText().
        LEAVE. 
      END.
    END.
  END.

  /* Suppress the pointless display of some AppBuilder comments, 
   * which are actually for the previous internal procedure. 
   */
  IF pComment = "/* _UIB-CODE-BLOCK-END */" 
  OR pComment = 
    "/* **********************  Internal Procedures  *********************** */"
  THEN 
    ASSIGN pComment = "". 

END PROCEDURE.




PROCEDURE getBlockComment:
/*
 * Procedure to find comments at the top of a Code_block
 * Input the parent of the Code_block node (i.e. PROCEDURE or FUNCTION, etc)
 */

  DEFINE INPUT  PARAMETER pNode    AS INTEGER   NO-UNDO. /* PROCEDURE/FUNCTION */
  DEFINE OUTPUT PARAMETER pComment AS CHARACTER NO-UNDO. /* text of comment */

  DEFINE VARIABLE blockNode  AS INTEGER NO-UNDO.
  DEFINE VARIABLE foundNode AS LOGICAL NO-UNDO.
  DEFINE VARIABLE realNode  AS INTEGER NO-UNDO.

  /* Find a "Code_block" child (synthetic node) */
  RUN findChild (INPUT pNode,
                 INPUT "Code_block":U,
                 OUTPUT blockNode).
  IF blockNode > 0 THEN DO:
    RUN getRealNode (INPUT blockNode, OUTPUT realNode, OUTPUT foundNode).
    IF foundNode THEN DO:
      RUN getCommentBefore (INPUT realNode, OUTPUT pComment).
      parserReleaseHandle(realNode).
    END.
    parserReleaseHandle(blockNode).
  END.

  RETURN.
END PROCEDURE. /* getBlockComment */




/* ***** John Green (TODO) hiddenGetAfter is no longer available *****
PROCEDURE getCommentAfter: 
 
/* Procedure to find a hidden comment token after the given node, and 
 * return its' text. 
 */

  DEFINE INPUT  PARAMETER pNode     AS INTEGER   NO-UNDO. /* node to look after */ 
  DEFINE INPUT  PARAMETER pSameLine AS LOGICAL   NO-UNDO. /* must be same line? */ 
  DEFINE OUTPUT PARAMETER pComment  AS CHARACTER NO-UNDO. /* text of comment */

  IF parserHiddenGetAfter(pNode) = TRUE THEN 
  find-block:
  DO: 

    /* If the comment has to be on the same line as the parameter, and 
     * the WS node contains a newline, the comment won't be on the same line
     * so we aren't going to find a suitable comment. 
     */
    IF  pSameLine = TRUE  
    AND parserHiddenGetType() = "WS" 
    AND INDEX(parserHiddenGetText(),"~n") <> 0 THEN LEAVE find-block.

    IF parserHiddenGetType() = "COMMENT" THEN
      ASSIGN pComment = parserHiddenGetText().

    ELSE 
    DO WHILE parserHiddenGetNext() = TRUE: 
      IF  pSameLine = TRUE  
      AND parserHiddenGetType() = "WS" 
      AND INDEX(parserHiddenGetText(),"~n") <> 0 THEN LEAVE find-block.

      IF parserHiddenGetType() = "COMMENT" THEN DO: 
        ASSIGN pComment = parserHiddenGetText().
        LEAVE find-block. 
      END.
    END.
  END.

  /* Suppress the pointless display of some AppBuilder comments. */
  IF pComment = "/* _UIB-CODE-BLOCK-END */"
  OR pComment = 
    "/* **********************  Internal Procedures  *********************** */"
  THEN 
    ASSIGN pComment = "". 

END PROCEDURE.
***** John Green (TODO) hiddenGetAfter is no longer available ***** */





PROCEDURE getRealNode:

/* Procedure to find the first real node after a given node. A real node
 * is one which has non-blank nodeText. 
 * Note that getRealNode is recursive,
 * and the "child" handle must be defined inside this procedure.
 */

  DEFINE INPUT  PARAMETER theNode   AS INTEGER NO-UNDO.
  DEFINE OUTPUT PARAMETER realNode  AS INTEGER NO-UNDO.
  DEFINE OUTPUT PARAMETER foundNode AS LOGICAL NO-UNDO.

  DEFINE VARIABLE child    AS INTEGER NO-UNDO.
  DEFINE VARIABLE haveNode AS LOGICAL NO-UNDO.

  ASSIGN 
    child    = parserGetHandle()
    haveNode = parserNodeFirstChild(theNode,child) <> "".

  DO WHILE haveNode:
    IF parserGetNodeText(child) <> "" THEN DO: 
      ASSIGN 
        realNode  = child
        foundNode = TRUE. 
      RETURN. 
    END. 
    /* If this is a new node head, run getRealNode with it */
    IF parserNodeFirstChild(child, grandchild) <> "" THEN DO:
      RUN getRealNode (INPUT child, OUTPUT realNode, OUTPUT foundNode).
      IF foundNode THEN RETURN. 
    END.
    ASSIGN haveNode = parserNodeNextSibling(child,child) <> "".
  END.
  parserReleaseHandle(child).

END PROCEDURE.




PROCEDURE findChild:
/* Does a "shallow" search for a child of the input node.
 * Returns a new handle if found, or 0 if not found.
 * If found, then calling procedure is responsible for deleting the handle.
 */
  DEFINE INPUT  PARAMETER theParent AS INTEGER NO-UNDO.
  DEFINE INPUT  PARAMETER findType  AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER retHandle AS INTEGER NO-UNDO.

  DEFINE VARIABLE currNode AS INTEGER NO-UNDO.
  DEFINE VARIABLE currNodeType AS CHARACTER NO-UNDO.
  ASSIGN currNode = parserGetHandle().

  currNodeType = parserNodeFirstChild(theParent, currNode).
  DO WHILE currNodeType > "" AND currNodeType <> findType:
    ASSIGN currNodeType = parserNodeNextSibling(currNode, currNode).
  END.

  IF currNodeType = findType THEN DO:
    retHandle = currNode.
  END.
  ELSE DO:
    parserReleaseHandle(currNode).
  END.

  RETURN.
END PROCEDURE. /* findChild */





PROCEDURE findNode:

/* Finds the first node of a particular type after a specified node.
 * Note that findNode is recursive, and the 
 * "child" handle must be defined inside this procedure.
 */

  DEFINE INPUT  PARAMETER theNode    AS INTEGER   NO-UNDO.
  DEFINE INPUT  PARAMETER findType   AS CHARACTER NO-UNDO. /* comma-delim type list */
  DEFINE OUTPUT PARAMETER resultNode AS INTEGER   NO-UNDO.
  DEFINE OUTPUT PARAMETER foundNode  AS LOGICAL   NO-UNDO.

  DEFINE VARIABLE child    AS INTEGER NO-UNDO.
  DEFINE VARIABLE havenode AS LOGICAL NO-UNDO.

  ASSIGN 
    child    = parserGetHandle()
    haveNode = parserNodeFirstChild(theNode,child) <> "".

  DO WHILE havenode:
    IF CAN-DO(findType, parserGetNodeType(child)) = TRUE THEN DO:
      ASSIGN  
        resultNode = child
        foundNode  = TRUE.
      RETURN. 
    END.

    /* If this is a new node head, run findNode with it */
    IF parserNodeFirstChild(child, grandchild) <> "" THEN DO:
      RUN findNode (INPUT  child, 
                    INPUT  findType, 
                    OUTPUT resultNode, 
                    OUTPUT foundNode).
      IF foundNode THEN RETURN. 
    END.
    ASSIGN haveNode = parserNodeNextSibling(child,child) <> "".
  END.
  parserReleaseHandle(child).

END PROCEDURE.





PROCEDURE getStmtText:

/* Hand in the node that you want the retrieved text to start at. 
 * This procedure ends at the first COLON (LEXCOLON) or PERIOD it gets to. 
 * Note that getStmtText is recursive,
 * and the "child" handle must be defined inside this procedure.
 */

  DEFINE INPUT        PARAMETER theNode AS INTEGER   NO-UNDO. /* start Node */
  DEFINE INPUT-OUTPUT PARAMETER theText AS CHARACTER NO-UNDO. 
  DEFINE OUTPUT       PARAMETER done    AS LOGICAL   NO-UNDO.

  DEFINE VARIABLE child    AS INTEGER NO-UNDO.
  DEFINE VARIABLE havenode AS LOGICAL NO-UNDO.

  ASSIGN 
    child    = parserGetHandle()
    theText  = theText + parserGetNodeText(theNode) + " "
    havenode = parserNodeFirstChild(theNode, child) <> "".

  DO WHILE havenode:
    IF CAN-DO("LEXCOLON,PERIOD", parserGetNodeType(child)) = TRUE THEN DO:
      ASSIGN 
        theText = theText + parserGetNodeText(child)
        done    = TRUE. 
      RETURN. 
    END.

    /* If this is a new node head, run getStmtText with it */
    IF parserNodeFirstChild(child, grandchild) <> "" THEN 
      RUN getStmtText (INPUT child, INPUT-OUTPUT theText, OUTPUT done).
      IF done THEN RETURN. 
    ELSE 
      ASSIGN theText = theText + parserGetNodeText(child) + " ".
    ASSIGN haveNode = parserNodeNextSibling(child,child) <> "".
  END.

  parserReleaseHandle(child).

END PROCEDURE.





PROCEDURE replaceChars:   /* Replace characters significant to HTML. */

  DEFINE INPUT-OUTPUT PARAMETER pText AS CHARACTER NO-UNDO. /* text to fix */
  IF pText = "" THEN RETURN.

  ASSIGN 
    pText = REPLACE(pText, "&","&amp;")       /* Do & first! */
    pText = REPLACE(pText, "<","&lt;")
    pText = REPLACE(pText, ">","&gt;").

END PROCEDURE.



PROCEDURE setFileList:   /* Sets filename containing list of programs to document. */
  DEFINE INPUT PARAMETER p AS CHARACTER NO-UNDO.
  ASSIGN fileList = p.
END PROCEDURE.



PROCEDURE setOutputDir:   /* Sets the output directory. */
  DEFINE INPUT PARAMETER dir AS CHARACTER NO-UNDO.
  ASSIGN outputDir = dir.
END.


 
PROCEDURE setAppName:   /* Sets the application name. */
  DEFINE INPUT PARAMETER name AS CHARACTER NO-UNDO.
  ASSIGN appName = name.
END.



PROCEDURE getNumErrors:    /* Reports the number of errors found. */
  DEFINE OUTPUT PARAMETER p AS INTEGER NO-UNDO.
  ASSIGN p = numErrors.
END.



PROCEDURE setProcCommentLoc:   /* Sets where to look for PROCEDURE comments. */
  DEFINE INPUT PARAMETER loc AS LOGICAL NO-UNDO.
  ASSIGN procCommentLoc = loc.
END.



PROCEDURE setFuncCommentLoc:   /* Sets where to look for FUNCTION comments. */
  DEFINE INPUT PARAMETER loc AS LOGICAL NO-UNDO.
  ASSIGN funcCommentLoc = loc.
END.



PROCEDURE setParamCommentLoc:  
  /* Sets where to look for DEFINE PARAMETER comments. */
  DEFINE INPUT PARAMETER loc AS LOGICAL NO-UNDO.
  ASSIGN paramCommentLoc = loc.
END.





PROCEDURE writeIndexFile: /* Write out the HTML to build the index html file. */

  OUTPUT STREAM Sind TO VALUE(outputDir + indFile).

  PUT STREAM Sind UNFORMATTED 
    '<HTML>~n'
    '<HEAD>~n'
    '<TITLE> '
    appName 
    ' </TITLE>~n'
    '</HEAD>~n~n'

    '<FRAMESET COLS="215,*">~n'
    '<FRAME NAME="Toc" SRC="'
    tocFile
    '" SCROLLING="auto">~n'
    '<FRAME NAME="Details" SRC="'
    sumFile 
    '" SCROLLING="auto">~n~n'

    '<NOFRAMES>~n'
    '<BODY>~n' 
    '<A HREF="'
    sumFile
    '">' appName ' Documentation</A>~n'
    '</BODY>~n' 
    '</NOFRAMES>~n~n'

    '</FRAMESET>~n'
    '</HTML>~n'.

  OUTPUT STREAM Sind CLOSE.
 
END PROCEDURE.  





PROCEDURE writeTocFileHeader: 

  /* Write header for table of contents HTML. (left side of frame) */

  PUT STREAM Stoc UNFORMATTED 
    '<HTML>~n'
    '<HEAD>~n'
    '<TITLE> '
    appName 
    ' </TITLE>~n'
    '</HEAD>~n~n'

    '<BODY>~n'
    '<B><A HREF="' sumFile '" target="Details"> Main Page </A></B>~n~n' 
    '<P>~n~n'.

END PROCEDURE.  





PROCEDURE checkTocDir: 

  /* Check if we need to change directory name in the TOC HTML. */

  IF prevDir <> currDir THEN 
    PUT STREAM Stoc UNFORMATTED 
      '<BR><B> ' currDir ' </B><BR>~n'. 

END PROCEDURE.  





PROCEDURE writeTocFileLine: 

  /* Write a line for table of contents HTML. (left side of frame) */

  PUT STREAM Stoc UNFORMATTED 
    '<A HREF="' detOutFile '" target="Details">'
    currFile '</A><BR>~n'.
  
END PROCEDURE.  





PROCEDURE writeTocFileFooter: 

  /* Write footer for table of contents HTML. (left side of frame) */

  PUT STREAM Stoc UNFORMATTED 
    '~n~n'
    '</HTML>~n'
    '</BODY>~n'.
  
END PROCEDURE.  





PROCEDURE writeSumFileHeader: 
 
  /* Write header for summary HTML page. (right side of frame) */
  
  PUT STREAM Ssum UNFORMATTED 
    '<HTML>~n'
    '<HEAD>~n'
    '<TITLE>' 
    appName 
    '</TITLE>~n'
    '</HEAD>~n~n'

    '<BODY>~n~n'

    '<TABLE WIDTH=100%>~n'
    '<TR>~n~n'
 
    '<TD><H1> ' 
    appName 
    ' </H1></TD>~n~n'
    
    '<TD align="right" valign="top">~n~n'

    '<SMALL>~n'
    'Generated '
    TODAY
    ' by: &nbsp;&nbsp;&nbsp;&nbsp;&nbsp; ~n'
    '</SMALL><BR>~n~n'

    '<A HREF="http://www.joanju.com/autodox" target=_self>~n'
    '<IMG src="autodoxsmall.jpg" alt="[Link to AutoDox Home]" border="0">~n~n'

    '</TD>~n~n'

    '</TR>~n'
    '</TABLE>~n~n'.

END PROCEDURE.  





PROCEDURE checkSumDir: 

  /* Check if we need to change directory name in the summary HTML. */

  IF prevDir <> currDir THEN 
    PUT STREAM Ssum UNFORMATTED 
      '<BR><B> ' currDir ' </B><BR>~n'. 

END PROCEDURE.  





PROCEDURE writeSumFileLine: 
 
  /* Write a line for the summary HTML. (right side of frame) */

  PUT STREAM Ssum UNFORMATTED 
    '<A HREF="' detOutFile '" target="Details">' 
    currFile '</A>~n'
    '&nbsp;&nbsp;&nbsp;'
    '<CODE>' sComment '</CODE><BR> ~n~n'.

END PROCEDURE.  





PROCEDURE writeSumFileFooter: 

  /* Write footer for summary HTML page. (right side of frame) */
  
  PUT STREAM Ssum UNFORMATTED 
    '</HTML>~n'
    '</BODY>~n'.
 
END PROCEDURE.  





PROCEDURE prepareDetHtml: 

  /* Prepare the header and footer HTML for the multiple detail files. */

  ASSIGN detHeader = 
    '<HTML>~n' + 
    '<HEAD>~n' + 
    '<TITLE> ' + 
    appName  + 
    ' </TITLE>~n' + 
    '</HEAD>~n~n' + 
    '<BODY>~n~n'  + 
  
    '<!-- Left spacing column -->~n' + 
    '<TABLE cellpadding=3>~n' + 
    '<TR>~n' + 
    '<TD valign="top" width=5>~n' + 
    '</TD>~n' + 
  
    '<!-- Middle column -->~n' + 
    '<TD valign="top">~n' + 

    '<A NAME="topofpage"></A>~n'.


  ASSIGN detFooter = 
    '<A HREF="#topofpage"> Top </A>~n' + 
    '</TD>~n' + 
    '<!-- Right column -->~n' + 
    '<TD valign="top" width=5>~n' + 
    '</TD>~n' + 
    '</TR>~n' + 
    '</TABLE>~n~n' + 
    '</HTML>~n' + 
    '</BODY>~n'.
  
END PROCEDURE.  





PROCEDURE writeDetFile:    

  /* Write out the HTML to build the individual detail HTML file. */

  /* Prepare the detail output stream and output the html header text. */
  OUTPUT STREAM Sdet TO VALUE(outputDir + detOutFile).
  PUT STREAM Sdet UNFORMATTED detHeader.
  
  
  PUT STREAM Sdet UNFORMATTED 
    '<H2><A HREF="' fileName '">' fileName '</A></H2>~n'.
 
  IF lComment <> "" THEN 
    PUT STREAM Sdet UNFORMATTED 
      '<PRE>~n' 
      lComment 
      '</PRE>~n'.
  ELSE
    PUT STREAM Sdet UNFORMATTED 
      '<BR>~n'.

  
  
  /* Write the program's parameters. */
  PUT STREAM Sdet UNFORMATTED 
    '<H3> Parameters:</H3>~n'
    '<UL><CODE>~n'.

  ASSIGN found = FALSE. 
  FOR EACH Tparam WHERE Tparam.procName = "":  /* alpha sort by type and name */
  
    ASSIGN found = TRUE. 
  
    PUT STREAM Sdet UNFORMATTED
      Tparam.paramType '&nbsp;'
      Tparam.paramName '&nbsp;'
      Tparam.dataType  '&nbsp;&nbsp;'
      Tparam.paramComment '<BR>~n'.  
  
  END. /* for each Tparam */
  
  IF found = FALSE THEN 
    PUT STREAM Sdet UNFORMATTED 'None ~n'.
  
  PUT STREAM Sdet UNFORMATTED 
    '</CODE></UL><BR>~n'.
  
  

  /* Write out summary links for procedures. */
  PUT STREAM Sdet UNFORMATTED 
    '<H3><A HREF="#procedures"> Procedures Summary: </A></H3>~n'
    '<UL>~n'.

  ASSIGN found = FALSE. 
  FOR EACH Tproc BREAK BY Tproc.procName: 

    ASSIGN found = TRUE. 

    PUT STREAM Sdet UNFORMATTED
      '<A HREF="#' Tproc.procName '">'     
      Tproc.procName 
      '</A>~n'.

    IF NOT LAST(Tproc.procName) THEN 
      PUT STREAM Sdet UNFORMATTED '&nbsp; &#149; &nbsp;'.   /* bullet */

  END. 

  IF found = FALSE THEN 
    PUT STREAM Sdet UNFORMATTED 'None ~n'.

  PUT STREAM Sdet UNFORMATTED 
    '</UL><BR>~n'.



  /* Write out summary links for functions. */
  PUT STREAM Sdet UNFORMATTED 
    '<H3><A HREF="#functions"> Functions Summary: </A></H3>~n'
    '<UL>~n'.

  ASSIGN found = FALSE. 
  FOR EACH Tfunc BREAK BY Tfunc.funcName: 

    ASSIGN found = TRUE. 

    PUT STREAM Sdet UNFORMATTED
      '<A HREF="#' Tfunc.funcName '">'      
      Tfunc.funcName 
      '</A>~n'.

    IF NOT LAST(Tfunc.funcName) THEN 
      PUT STREAM Sdet UNFORMATTED '&nbsp; &#149; &nbsp;'.   /* bullet */ 

  END. 
       
  IF found = FALSE THEN 
    PUT STREAM Sdet UNFORMATTED 'None ~n'.

  PUT STREAM Sdet UNFORMATTED 
    '</UL>~n'
    '<HR>~n'.

  
  
  /* Write out details of any internal procedures and their parameters. */
  PUT STREAM Sdet UNFORMATTED 
    '<H2><A NAME="procedures"> Procedures: </A></H2>~n'.
  
  ASSIGN found = FALSE. 
  FOR EACH Tproc:          /* Sorted alphabetically by procName */
  
    ASSIGN found = TRUE. 
  
    PUT STREAM Sdet UNFORMATTED
      '<B>'
      '<A NAME="' Tproc.procName '">' Tproc.procName '</A>'
      '</B>~n'
      '<UL>~n'.
  
    IF Tproc.procComment <> "" THEN 
      PUT STREAM Sdet UNFORMATTED
        '<PRE>~n'
        Tproc.procComment '~n'
        '</PRE>~n'.
  
    PUT STREAM Sdet UNFORMATTED '<CODE>~n'.
  
    /* Write the procedure's parameters. */
    FOR EACH Tparam WHERE Tparam.procName = Tproc.procName: /* alpha sort */

      PUT STREAM Sdet UNFORMATTED
        Tparam.paramType '&nbsp;'
        Tparam.paramName '&nbsp;'
        Tparam.dataType '&nbsp;&nbsp;~n'
        Tparam.paramComment '<BR>~n'.  

    END. /* for each Tparam */
  
    PUT STREAM Sdet UNFORMATTED 
      '</CODE>~n'
      '</UL>~n'.
  
  END. /* for each Tproc */
  
  IF found = FALSE THEN
    PUT STREAM Sdet UNFORMATTED '<CODE><UL> None </UL></CODE>~n'.
  
  
  
  /* Write out details of any functions. */
  PUT STREAM Sdet UNFORMATTED 
    '<HR><H2><A NAME="functions"> Functions: </A></H2>~n'.
  
  ASSIGN found = FALSE. 
  FOR EACH Tfunc:          /* Sorted alphabetically by funcName */
  
    ASSIGN found = TRUE. 

    PUT STREAM Sdet UNFORMATTED
      '<B>'
      '<A NAME="' Tfunc.funcName '">' Tfunc.funcName '</A>'
      '</B>~n'
      '<UL>~n'.
  
    IF Tfunc.funcComment <> "" THEN 
      PUT STREAM Sdet UNFORMATTED
        '<PRE>~n'
        Tfunc.funcComment '~n'
        '</PRE>~n'.
  
    PUT STREAM Sdet UNFORMATTED 
      '<CODE>~n'
      Tfunc.funcText '~n'
      '</CODE>~n'.
  
    PUT STREAM Sdet UNFORMATTED 
      '</UL>~n'.
  
  END. /* for each Tfunc */
  
  IF found = FALSE THEN
    PUT STREAM Sdet UNFORMATTED '<CODE><UL> None </UL></CODE>~n'.
  
  
  
  /* Finish this program's detail file. */
  PUT STREAM Sdet UNFORMATTED detFooter.
  OUTPUT STREAM Sdet CLOSE. 

END PROCEDURE.  



