/******************************************************************************

    Program:        XXPDF_INC.I

    Written By:     Gordon Campbell - PRO-SYS Consultants Ltd.
    Written On:     June 2002

    Description:    Contains function and variable definitions for
                    generating a PDF document from within Progress

    Note:           This can only be included once per program

    --------------------- Revision History ------------------

    Date:     Author        Change Description

    07/12/02  G Campbell    Initial Release

    09/05/02  G Campbell    Fixed issue regarding the first call to pdf_set_font
                              - wasn't actually setting the font, had to be
                                called twice before accepting changes
                            Fixed annoying 'rebuild' message
                              - caused by inappropriate offset values when
                                producing XREF table

    09/10/02  G Campbell    Due to previous XREF changes, the pdf_load_image
                            and pdf_load_images functions had to change also

    10/14/02  G Campbell    Changed the element setting functions to procedures
                              - older versions of Progress were reaching the
                                64K Segment issue.

    10/18/02  G Campbell    Added routine called pdf_replace_text and call from
                            appropriate text procedures.  Used to handle special
                            characters in text strings.

    10/22/02  G Campbell    As per Steven L. Jellin (sjellin@elcb.co.za)

                            Added two procedure pdf_reset_all and
                            pdf_reset_stream.

    11/04/02  G Campbell    As per Julian Lyndon-Smith (jmls@tessera.co.uk)

                            Modified the Font/Image import procedures to use
                            the OS-APPEND command.

    02/28/03  G Campbell    As per SourceForge Bug 694886 - Adam Davies
                            (adam.davies@travellink.com.au)

                            The base14 Font "Helvetica" is missing the width of
                            a character.  This means all lookups into the array
                            are out by one (after about "D" in the ASCII chars).
                            This causes calls to pdf_text_width to be quite a
                            long way out.

    02/28/03  G Campbell    As per SourceForge Bug 694888 - Adam Davies
                            (adam.davies@travellink.com.au)

                            Function pdf_text_width in XXPDF_INC.I is supposed to
                            return the width of the passed in text parameter
                            based on the current font.

                            It does this by summing each characters width as
                            defined in an Adobe AFM file for the font (or
                            hard-coded for base14 fonts), then dividing by 1000
                            and multiplyting by the point size.

                            However, rounding is done via INTEGER conversion
                            after the first division instead of after the
                            multiplication resulting in poor accuracy.

    02/28/03  G Campbell    As per SourceForge Bug 695454 - Steve Southwell
                            (ses@bravepointdallas.com)

                            In pdf_set_Orientation, the pdf stream name was
                            hard-coded to "Spdf.

    02/28/03  G Campbell    As per Steve Southwell
                           (ses@bravepointdallas.com)

                            Added procedure pdf_text_center.  This centers text
                            on a certain X,Y point

    03/02/03  G Campbell    As per Adam Davies
                           (ses@bravepointdallas.com)

                            Added function pdf_get_NumFittingChars.  This
                            function returns the index of the last character
                            that will fit into the specified width.  It does
                            this by summing each characters AFM width (as
                            specified in the tt_pdf_font.font_width array) and
                            comparing with the required width (converted into
                            these same units).

    03/12/03  G Campbell    Added procedure pdf_link to allow for creation
                            of document links.

    03/19/03  G Campbell    As per Steve Southwell
                            (ses@bravepointdallas.com)

                            Added procedure pdf_wrap_text to wrap text within
                            a specific column range. Returns a parameter that
                            tells you the last Y position used.

                            Added function pdf_get_wrap_length.
                            You can use the function to see how long a piece of
                            text WOULD be if you were to wrap it.

    03/29/03  G Campbell    As per Kim Allbritain
                            (kimatrr@tampabay.rr.com)

                            pdf_text_width was always returning 0 (zero) when a
                            loaded (non Base) font was being used.  To fix this
                            issue I now run the pdf_ParseAFMFile routine as soon
                            as pdf_load_font is called.  This also required
                            additional field requirements to TT_pdf_font.

    06/23/03  G Campbell    Added ability to define a Rotation Angle.  Angles
                            available are 0,90,180 and 270.  Also updated code
                            to ensure that TextX and TextY were being set
                            correctly whenever pdf_Text was being created (based
                            on current angle).  Reworked placement of routines.

    07/02/03  G Campbell    Updated pdf_new_page to include calls to reset
                            Angle and Text X/Y coordinates.

    08/18/03  G Campbell    Added pdf_set_BottomMargin and pdf_BottomMargin.
                            This allows us to define a bottom most point when
                            creating a document.  If any text goes below the
                            Bottom Margin value (eg: 50) a new page is
                            automatically created.  This is useful for
                            converting text documents that don't have any page
                            markers (eg: ASCII chr(12) ).

                            Also, modified pdf_set_TextY to use the BottomMargin
                            to determine if any text elements have (or are)
                            going below the Bottom Margin value.  If so, create
                            a new page.

    08/18/03  G Campbell    As per Mike Frampton
                            (MIkeF@sequoiains.com)

                            Added pdf_watermark procedure.  Allows you to create
                            a text watermark (eg: Invoice) that appears on the
                            first layer of the PDF --- below any rectangles,
                            images etc.  Use with caution as you may not get the
                            complete watermark if you overlay with images.

                            Sample Call:

                            RUN pdf_watermark ("Spdf",
                                               "SAMPLE REPORT",
                                               "Courier-Bold",
                                               34,
                                               .5,.5,.14,
                                               300,500).

    08/18/03  G Campbell    As per Herbert Bayer ( I think)
                            Herbert Bayer [Herbert.Bayer@bundrinno.de]

                            Added pdf_skipn.  This allows you to skip 'n' number
                            of lines.  Saves calling pdf_skip numerous times.

    08/25/03  G Campbell    Added pdf_TotalPages function.

                            This allows you to include the Total Number of Pages
                            for the document directly into your report.  For
                            instance, if you wanted to have 'Page n of x' as
                            your Page footer then you can implement this with
                            pdf_TotalPages(streamname).

    08/26/03  G Campbell    Added PageHeader functionality.

                            Similar to the PageFooter functionality previously
                            outlined but for the top of the page.

    08/26/03  G Campbell    Updated pdf_xref to use PUT CONTROL and CHR(13)
                            and CHR(10).

                            This is due to compatability issues between UNIX
                            and Windows implementations.  Each Xref line must
                            be a certain number of characters and if we use
                            SKIP it will differ based on OS.

    08/27/03  G Campbell    Updated pdf_line to include a decimal value for
                            the line thickness.

******************************************************************************/

/* The following defines are used to determine the JPEG Image Height and Width */
&GLOBAL-DEFINE M_SOF0  "0xC0"                /* Start Of Frame N */
&GLOBAL-DEFINE M_SOF1  "0xC1"                /* N indicates which compression process */
&GLOBAL-DEFINE M_SOF2  "0xC2"       /* Only SOF0-SOF2 are now in common use */
&GLOBAL-DEFINE M_SOF3  "0xC3"
&GLOBAL-DEFINE M_SOF5  "0xC5"                /* NB: codes C4 and CC are NOT SOF markers */
&GLOBAL-DEFINE M_SOF6  "0xC6"
&GLOBAL-DEFINE M_SOF7  "0xC7"
&GLOBAL-DEFINE M_SOF9  "0xC9"
&GLOBAL-DEFINE M_SOF10 "0xCA"
&GLOBAL-DEFINE M_SOF11 "0xCB"
&GLOBAL-DEFINE M_SOF13 "0xCD"
&GLOBAL-DEFINE M_SOF14 "0xCE"
&GLOBAL-DEFINE M_SOF15 "0xCF"
&GLOBAL-DEFINE M_SOI   "0xD8"                /* Start Of Image (beginning of datastream) */
&GLOBAL-DEFINE M_EOI   "0xD9"                /* End Of Image (end of datastream) */
&GLOBAL-DEFINE M_SOS   "0xDA"                /* Start Of Scan (begins compressed data) */
&GLOBAL-DEFINE M_APP0  "0xE0"                /* Application-specific marker, type N */
&GLOBAL-DEFINE M_APP12 "0xEC"                /* (we don't bother to list all 16 APPn's) */
&GLOBAL-DEFINE M_COM   "0xFE"                /* COMment */
&GLOBAL-DEFINE M_MARK  "0xFF"       /* Marker */

/* ---------------------------- Define TEMP-TABLES -------------------------
   The temp-tables are used to store the PDF streams and resources used when
   generating a PDF document */
DEFINE TEMP-TABLE TT_pdf_stream
    FIELD obj_stream    AS CHARACTER
    FIELD obj_file      AS CHARACTER.

/* The following temp-table is used to store/track parameters per stream */
DEFINE TEMP-TABLE TT_pdf_param
    FIELD obj_stream    AS CHARACTER
    FIELD obj_parameter AS CHARACTER
    FIELD obj_valid     AS CHARACTER
    FIELD obj_value     AS CHARACTER
INDEX obj_parameter AS PRIMARY
      obj_parameter
INDEX obj_stream
      obj_stream.

DEFINE TEMP-TABLE TT_pdf_error
  FIELD obj_stream  AS CHARACTER
  FIELD obj_func    AS CHARACTER FORMAT "x(20)"
  FIELD obj_error   AS CHARACTER FORMAT "x(40)".

/* The following temp-table is used to build a list of objects that will appear
   in the PDF document */
DEFINE TEMP-TABLE TT_pdf_object
  FIELD obj_stream  AS CHARACTER
  FIELD obj_nbr     AS INTEGER
  FIELD obj_desc    AS CHARACTER
  FIELD obj_offset  AS DECIMAL DECIMALS 0 FORMAT "9999999999"
  FIELD gen_nbr     AS INTEGER FORMAT "99999"
  FIELD obj_type    AS CHARACTER FORMAT "X"
INDEX obj_nbr AS PRIMARY
      obj_nbr.

/* The following temp-table is used to store the actual content that will appear
   in the PDF document */
DEFINE TEMP-TABLE TT_pdf_content
    FIELD obj_stream    AS CHARACTER
    FIELD obj_seq       AS INTEGER
    FIELD obj_type      AS CHARACTER
    FIELD obj_content   AS CHARACTER FORMAT "X(60)"
    FIELD obj_page      AS INTEGER
    FIELD obj_line      AS INTEGER
    FIELD obj_length    AS INTEGER
INDEX obj_seq AS PRIMARY
      obj_stream
      obj_seq
INDEX obj_page
      obj_page
INDEX obj_stream
      obj_stream
INDEX obj_type
      obj_type.

/* The following temp-table is used to track Document Information */
DEFINE TEMP-TABLE TT_pdf_info
    FIELD obj_stream    AS CHARACTER
    FIELD info_attr     AS CHARACTER
    FIELD info_value    AS CHARACTER.

/* The following temp-table is used to track Links (Annotations)
   loaded into a PDF stream */
DEFINE TEMP-TABLE TT_pdf_link
    FIELD obj_stream    AS CHARACTER
    FIELD link_color    AS CHARACTER
    FIELD link_name     AS CHARACTER
    FIELD link_page     AS INTEGER
    FIELD link_x        AS INTEGER
    FIELD link_y        AS INTEGER
    FIELD link_h        AS INTEGER
    FIELD link_w        AS INTEGER
    FIELD link_border   AS INTEGER
    FIELD link_style    AS CHARACTER
    FIELD link_obj      AS INTEGER.

/* The following temp-table is used to track Images loaded into a PDF stream */
DEFINE TEMP-TABLE TT_pdf_image
    FIELD obj_stream    AS CHARACTER
    FIELD image_name    AS CHARACTER
    FIELD image_file    AS CHARACTER
    FIELD image_tag     AS CHARACTER
    FIELD image_obj     AS INTEGER
    FIELD image_len     AS INTEGER
    FIELD image_h       AS INTEGER
    FIELD image_w       AS INTEGER.

/* The following temp-table is used to track Fonts loaded into a PDF stream */
DEFINE TEMP-TABLE TT_pdf_font
    FIELD obj_stream        AS CHARACTER
    FIELD font_name         AS CHARACTER
    FIELD font_file         AS CHARACTER
    FIELD font_afm          AS CHARACTER
    FIELD font_type         AS CHARACTER
    FIELD font_width        AS CHARACTER
    FIELD font_obj          AS INTEGER
    FIELD font_descr        AS INTEGER
    FIELD font_stream       AS INTEGER
    FIELD font_len          AS INTEGER
    FIELD font_tag          AS CHARACTER
    FIELD afm_ItalicAngle   AS INTEGER
    FIELD afm_Ascender      AS CHARACTER
    FIELD afm_Descender     AS CHARACTER
    FIELD afm_FontBBox      AS CHARACTER
    FIELD afm_FirstChar     AS CHARACTER
    FIELD afm_LastChar      AS CHARACTER
    FIELD afm_Widths        AS CHARACTER
    FIELD afm_IsFixedPitch  AS CHARACTER INIT "0"
    FIELD afm_flags         AS CHARACTER
INDEX font_name AS PRIMARY
      font_name
INDEX obj_stream
      obj_stream.

/* ---------------------- Define LOCAL VARIABLES -------------------------- */
DEFINE VARIABLE pdf_inc_ContentSequence AS INTEGER NO-UNDO.
DEFINE VARIABLE pdf_inc_ObjectSequence  AS INTEGER NO-UNDO.

/* The following variables are used to store the Image Height Width */
DEFINE VARIABLE pdf_width     AS INTEGER NO-UNDO.
DEFINE VARIABLE pdf_height    AS INTEGER NO-UNDO.

DEFINE VARIABLE pdf_ForFooter           AS LOGICAL NO-UNDO.
DEFINE VARIABLE pdf_CurrentLine         AS INTEGER NO-UNDO.

DEFINE VARIABLE pdf-Res-Object      AS INTEGER NO-UNDO.
DEFINE VARIABLE Vuse-font           AS CHARACTER NO-UNDO.
DEFINE VARIABLE pdf-Stream-Start    AS INTEGER NO-UNDO.
DEFINE VARIABLE pdf-Stream-End      AS INTEGER NO-UNDO.

DEFINE STREAM S_pdf_inc.
DEFINE STREAM S_pdf_inp.

DEFINE BUFFER B_TT_pdf_content FOR TT_pdf_content.
DEFINE BUFFER B_TT_pdf_param   FOR TT_pdf_param.

/* ---------------------------- Define FUNCTIONS -------------------------- */

/* The following functions are used to determine the Images Width/Height */

/* end of Functions used to determine Image Height/Width */

PROCEDURE pdf_error :
  DEFINE INPUT PARAMETER pdfStream     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFunction   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfError      AS CHARACTER NO-UNDO.

  CREATE TT_pdf_error.
  ASSIGN TT_pdf_error.obj_stream = pdfStream
         TT_pdf_error.obj_func   = pdfFunction
         TT_pdf_error.obj_error  = pdfError.

END. /* RUN pdf_error */

PROCEDURE pdf_new :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFileName AS CHARACTER NO-UNDO.

  IF INDEX(pdfStream, " ") > 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_new","Cannot have a space in the Stream Name!").
    RETURN.
  END.

  CREATE TT_pdf_stream.
  ASSIGN TT_pdf_stream.obj_stream = pdfStream
         TT_pdf_stream.obj_file   = pdfFileName.

  RUN pdf_LoadBase14 (pdfStream).
  RUN pdf_init_param (pdfStream).

END. /* pdf_new */

PROCEDURE pdf_set_TextRed :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS DECIMAL NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TextRed","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "TextRed" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "TextRed".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_TextRed */

PROCEDURE pdf_set_TextGreen :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS DECIMAL NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TextGreen","Cannot find Stream!").
    RETURN .
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "TextGreen" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "TextGreen".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_TextGreen */

PROCEDURE pdf_set_TextBlue :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS DECIMAL NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TextBlue","Cannot find Stream!").
    RETURN.
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "TextBlue" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "TextBlue".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_TextBlue */

PROCEDURE pdf_set_PageWidth :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_PageWidth","Cannot find Stream!").
    RETURN .
  END.

  IF pdfValue = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_PageWidth","Page Width cannot be zero!").
    RETURN.
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "PageWidth" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "PageWidth".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_PageWidth */

PROCEDURE pdf_set_PageHeight :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_PageHeight","Cannot find Stream!").
    RETURN .
  END.

  IF pdfValue = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_PageHeight","Page Height cannot be zero!").
    RETURN .
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "PageHeight" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "PageHeight".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_PageHeight */


PROCEDURE pdf_set_Page :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_page","Cannot find Stream!").
    RETURN.
  END.

  IF pdfValue = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_page","Value passed cannot be zero!").
    RETURN.
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "Page" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "Page".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_Page */

PROCEDURE pdf_set_TextX :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  def var iInttextX as int no-undo.
  def var iPDFAngle as int no-undo.

  DEFINE BUFFER L_TT_pdf_param FOR TT_pdf_param.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TextX","Cannot find Stream (" + pdfStream + ")!").
    RETURN.
  END.

  IF pdfValue < 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TextX","Value cannot be less than or equal to zero!").
    RETURN .
  END.

  FIND FIRST L_TT_pdf_param
       WHERE L_TT_pdf_param.obj_stream    = pdfStream
         AND L_TT_pdf_param.obj_parameter = "TextX" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL L_TT_pdf_param THEN DO:
    CREATE L_TT_pdf_param.
    ASSIGN L_TT_pdf_param.obj_stream    = pdfStream
           L_TT_pdf_param.obj_parameter = "TextX".
  END.

  run ip_pdf_textX(pdfstream, output iIntTextX).
  run ip_pdf_Angle(pdfstream,output iPDFAngle).
  CASE IPDFAngle:
    WHEN 0 THEN
      L_TT_pdf_param.obj_value = STRING(iIntTextX + pdfValue).
    WHEN 180 THEN
      L_TT_pdf_param.obj_value = STRING(iIntTextX - pdfValue).
    OTHERWISE
      L_TT_pdf_param.obj_value = STRING(pdfValue).
  END CASE.

END. /* pdf_set_TextX */

PROCEDURE pdf_set_TextY :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  def var iIntTextY     as int no-undo.
  def var iPDFAngle     as int no-undo.
  def var iBottomMargin as int no-undo.

  DEFINE BUFFER L_TT_pdf_param FOR TT_pdf_param.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TextY","Cannot find Stream!").
    RETURN.
  END.

  run ip_pdf_BottomMargin(pdfstream, output iBottomMargin).
  IF pdfValue <= iBottomMargin
  AND NOT pdf_ForFooter THEN DO:
    run ip_pdf_TextY(input pdfstream, output iIntTextY).
    RUN pdf_new_page(pdfStream).
    pdfValue = iIntTextY.
  END.

  FIND FIRST L_TT_pdf_param
       WHERE L_TT_pdf_param.obj_stream    = pdfStream
         AND L_TT_pdf_param.obj_parameter = "TextY" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL L_TT_pdf_param THEN DO:
    CREATE L_TT_pdf_param.
    ASSIGN L_TT_pdf_param.obj_stream    = pdfStream
           L_TT_pdf_param.obj_parameter = "TextY".
  END.

  run ip_pdf_TextY(input pdfstream, output iIntTextY).
  run ip_pdf_Angle(input pdfstream, output iPDFAngle).
  CASE iPDFAngle:
    WHEN 90 THEN
      L_TT_pdf_param.obj_value = STRING(iIntTextY - pdfValue).
    WHEN 270 THEN
      L_TT_pdf_param.obj_value = STRING(iIntTextY + pdfValue).
    OTHERWISE
      L_TT_pdf_param.obj_value = STRING(pdfValue).
  END CASE.

END. /* pdf_set_TextY */


PROCEDURE pdf_set_GraphicX :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_GraphicX","Cannot find Stream!").
    RETURN.
  END.

  IF pdfValue < 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_GraphicX","Value cannot be less than 0!").
    RETURN.
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "GraphicX" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "GraphicX".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_GraphicX */

PROCEDURE pdf_set_GraphicY :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_GraphicY","Cannot find Stream!").
    RETURN.
  END.

  IF pdfValue < 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_GraphicY","Value cannot be less than zero!").
    RETURN.
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "GraphicY" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "GraphicY".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_GraphicY */

PROCEDURE pdf_set_info :
  DEFINE INPUT PARAMETER pdfStream    AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfAttribute AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfvalue     AS CHARACTER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_info","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_option  AS CHARACTER NO-UNDO.
  L_Option = "Author,Creator,Producer,Keywords,Subject,Title".

  IF LOOKUP(pdfAttribute,L_option) = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_info","Invalid Attribute passed!").
    RETURN .
  END.

  IF NOT CAN-FIND( FIRST TT_pdf_info
                   WHERE TT_pdf_info.obj_stream = pdfStream
                     AND TT_pdf_info.info_attr  = pdfAttribute NO-LOCK)
  THEN DO:
    CREATE TT_pdf_info.
    ASSIGN TT_pdf_info.obj_stream = pdfStream
           TT_pdf_info.info_attr  = pdfAttribute.
  END.

  TT_pdf_info.info_value = pdfValue.

END. /* pdf_set_info */

PROCEDURE pdf_move_to :
  DEFINE INPUT PARAMETER pdfStream    AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfToX       AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfToY       AS INTEGER NO-UNDO.

  def var iPage as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_move_to","Cannot find Stream!").
    RETURN .
  END.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "GRAPHIC"
         TT_pdf_content.obj_content = STRING(pdfToX) + " " + STRING(pdfToY) + " m" + CHR(13).
         TT_pdf_content.obj_page    = iPage.

END. /* pdf_moveto */

PROCEDURE pdf_rect :
  DEFINE INPUT PARAMETER pdfStream    AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFromX     AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFromY     AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfWidth     AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfHeight    AS INTEGER NO-UNDO.

  def var iPage as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_rect","Cannot find Stream!").
    RETURN.
  END.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "GRAPHIC"
         TT_pdf_content.obj_content = STRING(pdfFromX) + " " + STRING(pdfFromY) + " "
                                    + STRING(pdfWidth) + " " + STRING(pdfHeight)
                                    + " re" + CHR(13)
                                    + "B" + CHR(13).
         TT_pdf_content.obj_page    = iPage.

  RUN pdf_set_GraphicY (pdfStream,pdfFromY + pdfHeight).
  RUN pdf_set_GraphicX (pdfStream, pdfFromX + pdfWidth).

END. /* pdf_rect */

PROCEDURE pdf_stroke_color :
  DEFINE INPUT PARAMETER pdfStream    AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRed       AS DECIMAL NO-UNDO.
  DEFINE INPUT PARAMETER pdfGreen     AS DECIMAL NO-UNDO.
  DEFINE INPUT PARAMETER pdfBlue      AS DECIMAL NO-UNDO.


  def var iPage as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_stroke_color","Cannot find Stream!").
    RETURN .
  END.

  pdfRed   = IF pdfRed < 0 THEN 0
             ELSE IF pdfRed > 1 THEN 1
             ELSE pdfRed.
  pdfGreen = IF pdfGreen < 0 THEN 0
             ELSE IF pdfGreen > 1 THEN 1
             ELSE pdfGreen.
  pdfBlue  = IF pdfBlue < 0 THEN 0
             ELSE IF pdfBlue > 1 THEN 1
             ELSE pdfBlue.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "GRAPHIC"
         TT_pdf_content.obj_content = " " + STRING(pdfRed) + " " + STRING(pdfGreen)
                                    + " " + STRING(pdfBlue) + " RG ".
         TT_pdf_content.obj_page    = iPage.

END. /* pdf_rect */

PROCEDURE pdf_stroke_fill :
  DEFINE INPUT PARAMETER pdfStream    AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRed       AS DECIMAL NO-UNDO.
  DEFINE INPUT PARAMETER pdfGreen     AS DECIMAL NO-UNDO.
  DEFINE INPUT PARAMETER pdfBlue      AS DECIMAL NO-UNDO.

  def var iPage as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_stroke_fill","Cannot find Stream!").
    RETURN .
  END.

  pdfRed   = IF pdfRed < 0 THEN 0
             ELSE IF pdfRed > 1 THEN 1
             ELSE pdfRed.
  pdfGreen = IF pdfGreen < 0 THEN 0
             ELSE IF pdfGreen > 1 THEN 1
             ELSE pdfGreen.
  pdfBlue  = IF pdfBlue < 0 THEN 0
             ELSE IF pdfBlue > 1 THEN 1
             ELSE pdfBlue.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "GRAPHIC"
         TT_pdf_content.obj_content = " " + STRING(pdfRed) + " " + STRING(pdfGreen)
                                    + " " + STRING(pdfBlue) + " rg ".
         TT_pdf_content.obj_page    = iPage.

END. /* pdf_rect */

PROCEDURE pdf_set_dash :
  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfOn      AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfOff     AS INTEGER NO-UNDO.

  def var iPage as int no-undo.

  IF pdfOn  < 0 THEN pdfOn = 1.
  IF pdfOff < 0 THEN pdfOff = 1.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_dash","Cannot find Stream!").
    RETURN .
  END.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "GRAPHIC"
         TT_pdf_content.obj_content = " [" + STRING(pdfOn) + " " + STRING(pdfOff)
                                    + "] 0  d ".
         TT_pdf_content.obj_page    = iPage.

END. /* pdf_set_dash */

PROCEDURE pdf_line :
  DEFINE INPUT PARAMETER pdfStream    AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFromX     AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFromY     AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfToX       AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfToY       AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfWeight    AS DECIMAL DECIMALS 1 NO-UNDO.

  def var iPage as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_line","Cannot find Stream!").
    RETURN.
  END.

  DEFINE VARIABLE L_FromY   AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_ToY     AS INTEGER NO-UNDO.

  ASSIGN L_FromY =  /* pdf_PageHeight - */ (pdfFromY) /*(pdfFromY + pdf_TopMargin) * -1 */
         L_ToY   =  /* pdf_PageHeight - */ (pdfToY ) /* (pdfToY + pdf_TopMargin) * -1 */.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "GRAPHIC"
         TT_pdf_content.obj_content = STRING(pdfWeight) + " w" + CHR(13)
                                    + STRING(pdfFromX) + " " + STRING(L_FromY) + " m" + CHR(13)
                                    + STRING(pdfToX) + " " + STRING(L_ToY) + " l" + CHR(13)
                                    + "S" + CHR(13).
         TT_pdf_content.obj_page    = iPage.

  RUN pdf_set_GraphicX(pdfStream, pdfToX ).
  RUN pdf_set_GraphicY(pdfstream, pdfToY).

END. /* pdf_line */

PROCEDURE pdf_watermark:
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFont     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfSize     AS INTEGER   NO-UNDO.
  DEFINE INPUT PARAMETER pdfRed      AS DECIMAL   NO-UNDO.
  DEFINE INPUT PARAMETER pdfGreen    AS DECIMAL   NO-UNDO.
  DEFINE INPUT PARAMETER pdfBlue     AS DECIMAL   NO-UNDO.
  DEFINE INPUT PARAMETER pdfX        AS INTEGER   NO-UNDO.
  DEFINE INPUT PARAMETER pdfY        AS INTEGER   NO-UNDO.

 def var iPage as int no-undo.

  DEFINE VARIABLE L_Font      AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_PointSize AS CHARACTER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text","Cannot find Stream!").
    RETURN .
  END.

  RUN pdf_replace_text (INPUT-OUTPUT pdfText).

  pdfRed   = IF pdfRed < 0 THEN 0
             ELSE IF pdfRed > 1 THEN 1
             ELSE pdfRed.
  pdfGreen = IF pdfGreen < 0 THEN 0
             ELSE IF pdfGreen > 1 THEN 1
             ELSE pdfGreen.
  pdfBlue  = IF pdfBlue < 0 THEN 0
             ELSE IF pdfBlue > 1 THEN 1
             ELSE pdfBlue.

  FIND FIRST TT_pdf_font WHERE TT_pdf_font.obj_stream = pdfStream
                           AND TT_pdf_font.font_name  = pdfFont
                           NO-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_font THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_font","Font has not been loaded!").
    RETURN .
  END.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "WATERMARK"
         TT_pdf_content.obj_content = "BT" + CHR(13)
                                    + STRING(TT_pdf_font.font_tag) + " " + STRING(pdfSize) +  " Tf" + CHR(13)
                                    + "1 0 0 1 " + STRING(pdfX) +  " "
                                                 + STRING(pdfY) + " Tm" + CHR(13)
                                    + " " + STRING(pdfRed) + " "
                                          + STRING(pdfGreen) + " "
                                          + STRING(pdfBlue) + " rg" + CHR(13)
                                    + "50 TL" + CHR(13)
                                    + "T* (" + pdfText  + ") Tj" + CHR(13)
                                    + "ET" + CHR(13)
         TT_pdf_content.obj_line    = pdf_CurrentLine.
         TT_pdf_content.obj_page    = iPage.

END. /* pdf_watermark */

PROCEDURE pdf_text :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText     AS CHARACTER NO-UNDO.


  DEFINE VARIABLE L_Width   AS INTEGER NO-UNDO.
  def    var      iPage     as int     no-undo.
  def    var      iPDFAngle as int     no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text","Cannot find Stream!").
    RETURN .
  END.

  RUN pdf_replace_text (INPUT-OUTPUT pdfText).

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "TEXT"
         TT_pdf_content.obj_content = "(" + pdfText + ") Tj"
         TT_pdf_content.obj_line    = pdf_CurrentLine.
         TT_pdf_content.obj_page    = iPage.

  run ip_pdf_text_width(pdfStream, pdfText, output L_Width).
  run ip_pdf_Angle(pdfStream, pdfText, output iPDFAngle).
  CASE iPDFAngle:
    WHEN 0 OR WHEN 180 THEN
      RUN pdf_set_TextX( pdfStream, L_width ).
    WHEN 90 OR WHEN 270 THEN
      RUN pdf_set_TextY( pdfStream, L_width ).
  END CASE.

END. /* pdf_text */

PROCEDURE pdf_text_rotate :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfAngle    AS INTEGER NO-UNDO.

  DEFINE VARIABLE L_Rotate           AS CHARACTER NO-UNDO.
  def    var      iIntTextX          as int       no-undo.
  def    var      iIntTextY          as int       no-undo.
  def    var      iPage              as int       no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text","Cannot find Stream!").
    RETURN .
  END.

  CASE pdfAngle:
    WHEN 0 THEN
      L_Rotate = "1 0 0 1 ".
    WHEN 90 THEN
      L_Rotate = "0 -1 1 0 ".
    WHEN 180 THEN
      L_Rotate = "-1 0 0 -1 ".
    WHEN 270 THEN
      L_Rotate = "0 1 -1 0 ".
  END CASE.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_textX(pdfstream, output iIntTextX).
  run ip_pdf_TextY(input pdfstream, output iIntTextY).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "TEXT".
         TT_pdf_content.obj_content = L_Rotate + STRING(iIntTextX)
                                    + " " + STRING(iIntTextY) + " Tm".
  ASSIGN TT_pdf_content.obj_line    = pdf_CurrentLine.
         TT_pdf_content.obj_page    = iPage.

  RUN pdf_set_angle(pdfstream, pdfAngle).

END. /* pdf_text_rotate */

PROCEDURE pdf_set_font :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFont     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfSize     AS DECIMAL NO-UNDO.

  def var dpdfPointSize as dec no-undo.
  def var iPage         as int no-undo.
  def var iLeftMargin   as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_font","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_PointSize   AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_LeftMargin  AS CHARACTER NO-UNDO.
  run ip_pdf_PointSize(pdfStream, output dpdfPointSize).
  run ip_pdf_LeftMargin(pdfStream, output iLeftMargin).
  L_PointSize  = STRING(dpdfPointSize).
  L_LeftMargin = STRING(iLeftMargin).

  FIND FIRST TT_pdf_font WHERE TT_pdf_font.obj_stream = pdfStream
                           AND TT_pdf_font.font_name = pdfFont
                           NO-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_font THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_font","Font has not been loaded!").
    RETURN .
  END.

  run ip_pdf_page(pdfstream, output iPage).
  FIND FIRST B_TT_pdf_content
       WHERE B_TT_pdf_content.obj_stream = pdfStream
         AND B_TT_pdf_content.obj_page   = iPage
         AND B_TT_pdf_content.obj_line   = pdf_CurrentLine NO-ERROR.
  IF AVAIL B_TT_pdf_content THEN DO:
    ASSIGN B_TT_pdf_content.obj_content = B_TT_pdf_content.obj_content
                                        + CHR(13) + TT_pdf_font.font_tag + " "
                                        + STRING(pdfSize) +  " Tf" + CHR(13).
  END.

  ELSE DO:
    run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
    CREATE TT_pdf_content.
    ASSIGN TT_pdf_content.obj_stream  = pdfStream.
           TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
           TT_pdf_content.obj_type    = "TEXTFONT".
           TT_pdf_content.obj_line    = pdf_CurrentLine.
           TT_pdf_content.obj_page    = iPage.
           TT_pdf_content.obj_content =
                                      TT_pdf_font.font_tag + " "
                                      + STRING(pdfSize) +  " Tf".

  END.

  /* Set the Stream Font Parameter */
  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "Font" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "Font".
  END.

  TT_pdf_param.obj_value = pdfFont.

  /* Set the Stream Font Size */
  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "PointSize" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "PointSize".
  END.
  TT_pdf_param.obj_value = STRING(pdfSize).

END. /* pdf_set_font*/

PROCEDURE pdf_text_render :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRender   AS INTEGER NO-UNDO.

  def var iPage      as int no-undo.
  def var ipdfRender as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_render","Cannot find Stream!").
    RETURN .
  END.

  run ip_pdfRender(pdfstream, output ipdfRender).
  IF ipdfRender < 0 OR ipdfRender > 3 THEN ipdfRender = 0.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "TEXTRENDER"
         TT_pdf_content.obj_content = STRING(ipdfRender) +  " Tr"
         TT_pdf_content.obj_line    = pdf_CurrentLine.
         TT_pdf_content.obj_page    = iPage.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "Render" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "Render".
  END.

  TT_pdf_param.obj_value = STRING(ipdfRender).

END. /* pdf_text_render */

PROCEDURE pdf_text_color :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRed      AS DECIMAL NO-UNDO.
  DEFINE INPUT PARAMETER pdfGreen    AS DECIMAL NO-UNDO.
  DEFINE INPUT PARAMETER pdfBlue     AS DECIMAL NO-UNDO.

  def var dpdfPointSize as dec no-undo.
  def var iPage         as int no-undo.
  def var iLeftMargin   as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_color","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_PointSize   AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_LeftMargin  AS CHARACTER NO-UNDO.
  run ip_pdf_PointSize(pdfStream, output dpdfPointSize).
  run ip_pdf_LeftMargin(pdfStream, output iLeftMargin).
  L_PointSize  = STRING(dpdfPointSize).
  L_LeftMargin = STRING(iLeftMargin).

  run ip_pdf_page(pdfstream, output iPage).
  FIND FIRST B_TT_pdf_content
       WHERE B_TT_pdf_content.obj_stream = pdfStream
         AND B_TT_pdf_content.obj_page   = iPage
         AND B_TT_pdf_content.obj_line   = pdf_CurrentLine NO-ERROR.
  IF AVAIL B_TT_pdf_content THEN DO:
    ASSIGN B_TT_pdf_content.obj_content = B_TT_pdf_content.obj_content + CHR(13)
                                        + STRING(pdfRed) + " " + STRING(pdfGreen)
                                        + " " + STRING(pdfBlue) + " rg ".
  END.
  ELSE DO:
    run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
    CREATE TT_pdf_content.
    ASSIGN TT_pdf_content.obj_stream  = pdfStream.
           TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
    ASSIGN TT_pdf_content.obj_type    = "TEXTCOLOR"
           TT_pdf_content.obj_line    = pdf_CurrentLine.
           TT_pdf_content.obj_page    = iPage.
           TT_pdf_content.obj_content = /* "1 0 0 1 " + L_LeftMargin + " "
                                      + STRING( pdf_CurrentLine ) + " Tm" + CHR(13)
                                      + */ STRING(pdfRed) + " " + STRING(pdfGreen)
                                      + " " + STRING(pdfBlue) + " rg ".
  END.

  RUN pdf_set_TextRed(pdfStream, pdfRed).
  RUN pdf_set_TextGreen(pdfStream, pdfGreen).
  RUN pdf_set_TextBlue(pdfStream, pdfBlue).

END. /* pdf_text_color */

PROCEDURE pdf_load_font :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFontName AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFontFile AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFontAFM  AS CHARACTER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_load_font","Cannot find Stream!").
    RETURN .
  END.

  IF INDEX(pdfFontName," ") > 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_load_font","Font Name cannot contains spaces!").
    RETURN .
  END.

  IF SEARCH(pdfFontFile) = ? THEN DO:
    RUN pdf_error(pdfStream,"pdf_load_font","Cannot find Font File for Loading!").
    RETURN .
  END.

  IF SEARCH(pdfFontAFM) = ? THEN DO:
    RUN pdf_error(pdfStream,"ObjectSequence","Cannot find Font AFM file for loading!").
    RETURN .
  END.

  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.obj_stream  = pdfStream
         TT_pdf_font.font_name   = pdfFontName
         TT_pdf_font.font_file   = pdfFontFile
         TT_pdf_font.font_afm    = pdfFontAFM.
  TT_pdf_font.font_tag    = "/" + STRING(TT_pdf_font.font_name).

  RUN pdf_ParseAFMFile
      (OUTPUT TT_pdf_font.afm_ItalicAngle,
       OUTPUT TT_pdf_font.afm_Ascender,
       OUTPUT TT_pdf_font.afm_Descender,
       OUTPUT TT_pdf_font.afm_FontBBox,
       OUTPUT TT_pdf_font.afm_FirstChar,
       OUTPUT TT_pdf_font.afm_LastChar,
       OUTPUT TT_pdf_font.afm_Widths,
       OUTPUT TT_pdf_font.afm_IsFixedPitch,
       OUTPUT TT_pdf_font.afm_Flags) NO-ERROR.

      /* igc - Added March 29, 2003
           - need to assign l_afm_width to TT_pdf_font.font_width so that we
             can find the actual width when using pdf_text_width */
   TT_pdf_font.font_width = TT_pdf_font.afm_widths.

END. /* pdf_load_font */

PROCEDURE pdf_load_image :
  DEFINE INPUT PARAMETER pdfStream      AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfImageName   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfImageFile   AS CHARACTER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_load_image","Cannot find Stream!").
    RETURN .
  END.

  IF INDEX(pdfImageName," ") > 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_load_image","Image Name cannot contain spaces!").
    RETURN .
  END.


  IF SEARCH(pdfImageFile) = ? THEN DO:
    RUN pdf_error(pdfStream,"pdf_load_image","Cannot find Image File when Loading!").
    RETURN .
  END.

  RUN pdf_get_image_wh (INPUT pdfStream,
                        INPUT pdfImageFile).


  CREATE TT_pdf_image.
  ASSIGN TT_pdf_image.obj_stream    = pdfStream
         TT_pdf_image.image_name    = pdfImageName
         TT_pdf_image.image_file    = pdfImageFile
         TT_pdf_image.image_h       = pdf_Height
         TT_pdf_image.image_w       = pdf_Width.
  TT_pdf_image.image_tag    = "/Im" + STRING(TT_pdf_image.image_name).

END. /* pdf_load_image */

PROCEDURE pdf_place_image :
  DEFINE INPUT PARAMETER pdfStream    AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfImageName AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfColumn    AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRow       AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfWidth     AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfHeight    AS INTEGER NO-UNDO.

  def var iPage as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_place_image","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_PageHeight  AS INTEGER NO-UNDO.
  run ip_pdf_PageHeight(pdfstream, output L_PageHeight).

  FIND FIRST TT_pdf_image
       WHERE TT_pdf_image.image_name = pdfImageName NO-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_image THEN DO:
    RUN pdf_error(pdfStream,"pdf_place_image","Cannot find Image Name for Placement!").
    RETURN .
  END.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "IMAGE"
         TT_pdf_content.obj_content = STRING(pdfWidth) + " 0 0 " + STRING(pdfHeight)
                                    + " " + STRING(pdfColumn) + " "
                                    + STRING(L_PageHeight - pdfRow) + " cm "
                                    + TT_pdf_image.image_tag + " Do" + CHR(13).
         TT_pdf_content.obj_page    = iPage.

  RUN pdf_set_GraphicX(pdfStream,pdfColumn).
  RUN pdf_set_GraphicY(pdfStream,pdfRow).

END. /* pdf_place_image */


PROCEDURE pdf_skip :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.

  def var iIntTextY   as int no-undo.
  def var iIntVert    as int no-undo.
  def var iPage       as int no-undo.
  def var iLeftMargin as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_skip","Cannot find Stream!").
    RETURN .
  END.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "TEXTSKIP"
         TT_pdf_content.obj_content = "T*"
         TT_pdf_content.obj_line    = pdf_CurrentLine.
         TT_pdf_content.obj_page    = iPage.

  run ip_pdf_VerticalSpace(input pdfStream, output iIntVert).
  run ip_pdf_LeftMargin(input pdfStream, output iLeftMargin).
  pdf_CurrentLine  = pdf_CurrentLine - iIntVert.
  RUN pdf_set_TextX (pdfStream, iLeftmargin).
  run ip_pdf_TextY(input pdfstream, output iIntTextY).
  RUN pdf_set_TextY (pdfStream, (iIntTextY - iIntVert)).

END. /* pdf_skip */

PROCEDURE pdf_skipn :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfNumber   AS INTEGER NO-UNDO.

  DEFINE VARIABLE l_Ctr              AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_skipn","Cannot find Stream!").
    RETURN .
  END.

  IF pdfNumber <= 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_skipn","Lines to skip cannot be <= zero!").
    RETURN .
  END.

  DO l_Ctr = 1 TO pdfNumber:
    RUN pdf_skip(pdfStream).
  END.

END. /* pdf_skipn */

PROCEDURE pdf_text_xy :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfColumn   AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRow      AS INTEGER NO-UNDO.

  def var dpdfPointSize as dec no-undo.
  def var dTextRed      as dec no-undo.
  def var dTextGreen    as dec no-undo.
  def var dTextBlue     as dec no-undo.
  def var iPage         as int no-undo.
  def var iLeftMargin   as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_xy","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_color   AS CHARACTER NO-UNDO.

  run ip_pdf_TextRed(pdfstream, output dTextRed).
  run ip_pdf_TextGreen(pdfstream, output dTextGreen).
  run ip_pdf_TextBlue(pdfstream, output dTextBlue).

  L_color = STRING(dTextRed) + " "
          + STRING(dTextGreen) + " "
          + STRING(dTextBlue) + " rg" + CHR(13).

  DEFINE VARIABLE L_PointSize   AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_LeftMargin  AS CHARACTER NO-UNDO.
  run ip_pdf_PointSize(pdfStream, output dpdfPointSize).
  run ip_pdf_LeftMargin(pdfStream, output iLeftMargin).
  L_PointSize  = STRING(dpdfPointSize).
  L_LeftMargin = STRING(iLeftMargin).

  DEFINE VARIABLE L_Font        AS CHARACTER NO-UNDO.
  run ip_pdf_Font(input pdfstream, output L_font).

  FIND FIRST TT_pdf_font
       WHERE TT_pdf_font.obj_stream = pdfStream
         AND TT_pdf_font.font_name  = L_Font NO-LOCK NO-ERROR.
  L_Font = IF AVAIL TT_pdf_font THEN TT_pdf_Font.font_tag ELSE "/F1".

  DEFINE VARIABLE L_orig_text   AS CHARACTER NO-UNDO.
  L_orig_text = pdfText.

  IF pdfColumn = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_xy","Column cannot be zero!").
    RETURN .
  END.

  IF pdfRow    = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_xy","Row cannot be zero!").
    RETURN .
  END.

  RUN pdf_replace_text (INPUT-OUTPUT pdfText).

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "TEXTXY"
         TT_pdf_content.obj_line    = pdfRow
         TT_pdf_content.obj_content = /* "BT" + CHR(13) */
                                    L_Font + " " + L_PointSize + " Tf" + CHR(13)
                                    + L_color
                                    + "1 0 0 1 " + STRING(pdfColumn) + " "
                                    + STRING(pdfRow) + " Tm" + CHR(13)
                                    + L_LeftMargin + " TL" + CHR(13)
                                    + "(" + pdfText + ") Tj" + CHR(13)
                                    /* + "ET" + CHR(13) */
         TT_pdf_content.obj_length  = LENGTH( L_orig_text) .
         TT_pdf_content.obj_page    = iPage.

  RUN pdf_set_textY(pdfStream, pdfRow).

END. /* pdf_text_xy */

PROCEDURE pdf_text_boxed_xy :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfColumn   AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRow      AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfWidth    AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfHeight   AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfJustify  AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfWeight   AS INTEGER NO-UNDO.

  def var dpdfPointSize as dec no-undo.
  def var dTextRed      as dec no-undo.
  def var dTextGreen    as dec no-undo.
  def var dTextBlue     as dec no-undo.
  def var iPage         as int no-undo.
  def var iLeftMargin   as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_boxed_xy","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_color   AS CHARACTER NO-UNDO.

  run ip_pdf_textred(pdfstream, output dTextRed).
  run ip_pdf_TextGreen(pdfstream, output dTextGreen).
  run ip_pdf_TextGreen(pdfstream, output dTextGreen).

  L_color = STRING(dTextRed) + " "
          + STRING(dTextGreen) + " "
          + STRING(dTextBlue) + " rg" + CHR(13).

  DEFINE VARIABLE L_PointSize   AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_LeftMargin  AS CHARACTER NO-UNDO.
  run ip_pdf_PointSize(pdfStream, output dpdfPointSize).
  run ip_pdf_LeftMargin(pdfStream, output iLeftMargin).
  L_PointSize  = STRING(dpdfPointSize).
  L_LeftMargin = STRING(iLeftMargin).

  DEFINE VARIABLE L_Font        AS CHARACTER NO-UNDO.
  run ip_pdf_Font(input pdfstream, output L_Font).
  FIND FIRST TT_pdf_font
       WHERE TT_pdf_font.obj_stream = pdfStream
         AND TT_pdf_font.font_name  = L_Font
         NO-LOCK NO-ERROR.
  L_Font = IF AVAIL TT_pdf_font THEN TT_pdf_Font.font_tag ELSE "/F1".

  DEFINE VARIABLE L_orig_text   AS CHARACTER NO-UNDO.
  L_orig_text = pdfText.

  IF pdfColumn = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_boxed_xy","Column cannot be zero!").
    RETURN .
  END.

  IF pdfRow    = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_boxed_xy","Row cannot be zero!").
    RETURN .
  END.

  IF pdfHeight = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_boxed_xy","Height cannot be zero!").
    RETURN .
  END.

  IF pdfWidth  = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_boxed_xy","Width cannot be zero!").
    RETURN .
  END.

  IF LOOKUP(pdfJustify,"Left,Right,Center") = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_boxed_xy","Invalid Justification option passed!").
    RETURN .
  END.

  RUN pdf_replace_text (INPUT-OUTPUT pdfText).

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "TEXTXY"
         TT_pdf_content.obj_line    = pdfRow
         TT_pdf_content.obj_content = /* "BT" + CHR(13)
                                    + */ L_Font + " " + L_PointSize + " Tf" + CHR(13)
                                    + L_color
                                    + "1 0 0 1 " + STRING(pdfColumn) + " "
                                    + STRING(pdfRow) + " Tm" + CHR(13)
                                    + L_LeftMargin + " TL" + CHR(13)
                                    + "(" + pdfText + ") Tj" + CHR(13)
                                    /* + "ET" + CHR(13) */
         TT_pdf_content.obj_length  = LENGTH( L_orig_text).
         TT_pdf_content.obj_page    = iPage.

  IF pdfWeight = 0 THEN pdfWeight = 1.

  run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
  ASSIGN TT_pdf_content.obj_type    = "GRAPHIC"
         TT_pdf_content.obj_line    = pdfRow
         TT_pdf_content.obj_content = "q" + CHR(13)
                                    + STRING(pdfWeight) + " w" + CHR(13)
                                    + STRING(pdfColumn) + " " + STRING(pdfRow) + " "
                                    + STRING(pdfWidth) + " " + STRING(pdfHeight)
                                    + " re" + CHR(13)
                                    + "B" + CHR(13)
                                    + "Q" + CHR(13)
         TT_pdf_content.obj_length  = LENGTH( L_orig_text) .
         TT_pdf_content.obj_page    = iPage.

  RUN pdf_set_TextY(pdfStream, pdfRow).
  RUN pdf_set_GraphicY(pdfStream, pdfRow).

END. /* pdf_text_boxed_xy */

PROCEDURE pdf_text_center:
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfColumn   AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRow      AS INTEGER NO-UNDO.

  def var iWidth as int no-undo.

  run ip_pdf_text_width (pdfStream, pdfText, output iWidth).
  RUN pdf_text_xy (pdfStream,pdfText,pdfColumn
                   - INTEGER(iWidth / 2),pdfRow).
END. /* pdf_text_center */

PROCEDURE pdf_text_at :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfColumn   AS INTEGER NO-UNDO.

  def var iIntTextY   as int no-undo.
  def var iPage       as int no-undo.
  def var iLeftMargin as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_at","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_TextY       AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_LeftMargin  AS CHARACTER NO-UNDO.
  run ip_pdf_TextY(input pdfstream, output L_TextY).
  run ip_pdf_LeftMargin(input pdfstream, output iLeftMargin).
  L_LeftMargin = STRING(iLeftMargin).

  DEFINE VARIABLE L_orig_text   AS CHARACTER NO-UNDO.
  L_orig_text = pdfText.

  IF pdfColumn = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_at","Column cannot be zero!").
    RETURN .
  END.

  RUN pdf_replace_text (INPUT-OUTPUT pdfText).

  run ip_pdf_TextY(input pdfstream, output iIntTextY).
  run ip_pdf_page(pdfstream, output iPage).

  FIND FIRST B_TT_pdf_content
       WHERE B_TT_pdf_content.obj_stream = pdfStream
         AND B_TT_pdf_content.obj_page   = iPage
         AND B_TT_pdf_content.obj_line   = IIntTextY NO-ERROR.
  IF AVAIL B_TT_pdf_content THEN DO:
    ASSIGN B_TT_pdf_content.obj_content = B_TT_pdf_content.obj_content
                                        + " "
                                        + "(" + FILL(" ", pdfColumn - B_TT_pdf_content.obj_length - 1)
                                        + pdfText + ") Tj "
           B_TT_pdf_content.obj_length  = B_TT_pdf_content.obj_length
                                        + LENGTH(FILL(" ", pdfColumn - B_TT_pdf_content.obj_length - 1) + L_orig_text).
  END.
  ELSE DO:
    run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
    run ip_pdf_TextY(input pdfstream, output iIntTextY).

    CREATE TT_pdf_content.
    ASSIGN TT_pdf_content.obj_stream  = pdfStream.
           TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
    ASSIGN TT_pdf_content.obj_type    = "TEXTAT".
           TT_pdf_content.obj_line    = iIntTextY.
           TT_pdf_content.obj_page    = iPage.
    ASSIGN TT_pdf_content.obj_content = IF NOT pdf_ForFooter THEN
                                          "1 0 0 1 " + L_LeftMargin + " "
                                          + L_TextY + " Tm" + CHR(13)
                                          + "(" + FILL(" ", pdfColumn - 1) + pdfText + ") Tj "
                                        ELSE
                                          "(" + FILL(" ", pdfColumn - 1) + pdfText + ") Tj "
           TT_pdf_content.obj_length  = LENGTH( FILL(" ",pdfColumn - 1) + L_orig_text) .
  END.

END. /* pdf_text_at */

PROCEDURE pdf_text_to :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfColumn   AS INTEGER NO-UNDO.

  def var iIntTextY   as int no-undo.
  def var iPage       as int no-undo.
  def var iLeftMargin as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_to","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_TextY       AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_LeftMargin  AS CHARACTER NO-UNDO.
  run ip_pdf_TextY(input pdfstream, output L_TextY).
  run ip_pdf_LeftMargin(pdfstream, output iLeftMargin).
  L_LeftMargin = STRING(iLeftMargin).

  DEFINE VARIABLE L_orig_text   AS CHARACTER NO-UNDO.
  L_orig_text = pdfText.

  IF pdfColumn = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_text_to","Column cannot be zero!").
    RETURN .
  END.

  RUN pdf_replace_text (INPUT-OUTPUT pdfText).

  run ip_pdf_TextY(input pdfstream, output iIntTextY).
  run ip_pdf_page(pdfstream, output iPage).
  FIND FIRST B_TT_pdf_content
       WHERE B_TT_pdf_content.obj_stream = pdfStream
         AND B_TT_pdf_content.obj_page   = iPage
         AND B_TT_pdf_content.obj_line   = iIntTextY NO-ERROR.
  IF AVAIL B_TT_pdf_content THEN DO:
    ASSIGN B_TT_pdf_content.obj_content = B_TT_pdf_content.obj_content
                                        + " "
                                        + "(" + FILL(" ", pdfColumn - (B_TT_pdf_content.obj_length + LENGTH(L_Orig_text)))
                                        + pdfText + ") Tj "
           B_TT_pdf_content.obj_length  = B_TT_pdf_content.obj_length
                                        + LENGTH(FILL(" ", pdfColumn - (B_TT_pdf_content.obj_length + LENGTH(L_orig_text))) + L_orig_text).
  END.
  ELSE DO:
    run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
    run ip_pdf_TextY(input pdfstream, output iIntTextY).

    CREATE TT_pdf_content.
    ASSIGN TT_pdf_content.obj_stream  = pdfStream.
           TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
    ASSIGN TT_pdf_content.obj_type    = "TEXTAT"
           TT_pdf_content.obj_content = IF NOT pdf_ForFooter THEN
                                          "1 0 0 1 " + L_LeftMargin + " "
                                          + L_TextY + " Tm" + CHR(13)
                                          + "(" + FILL(" ", pdfColumn - LENGTH( L_orig_text) - 1) + pdfText + ") Tj "
                                        ELSE
                                          "(" + FILL(" ", pdfColumn - LENGTH( L_orig_text) - 1) + pdfText + ") Tj "
           TT_pdf_content.obj_length  = LENGTH( FILL(" ",pdfColumn - LENGTH(L_orig_text) - 1) + L_orig_text) .
           TT_pdf_content.obj_line    = iIntTextY.
           TT_pdf_content.obj_page    = iPage.
  END.

END. /* pdf_text_to */


PROCEDURE pdf_set_Angle :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_Angle","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_option  AS CHARACTER NO-UNDO.
  L_Option = "0,90,180,270".

  IF LOOKUP(STRING(pdfValue),L_option) = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_Angle","Invalid Angle option passed!").
    RETURN .
  END.

  FIND FIRST B_TT_pdf_param
       WHERE B_TT_pdf_param.obj_stream    = pdfStream
         AND B_TT_pdf_param.obj_parameter = "Angle" NO-LOCK NO-ERROR.
  IF NOT AVAIL B_TT_pdf_param THEN DO:
    CREATE B_TT_pdf_param.
    ASSIGN B_TT_pdf_param.obj_stream    = pdfStream
           B_TT_pdf_param.obj_parameter = "Angle".
  END.

  B_TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_Angle */

PROCEDURE pdf_set_Orientation :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_width   AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_height  AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_Orientation","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_option  AS CHARACTER NO-UNDO.
  L_Option = "Portrait,Landscape".

  IF LOOKUP(pdfValue,L_option) = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_Orientation","Invalid Orientation option passed!").
    RETURN .
  END.

  FIND FIRST B_TT_pdf_param
       WHERE B_TT_pdf_param.obj_stream    = pdfStream
         AND B_TT_pdf_param.obj_parameter = "Orientation" NO-LOCK NO-ERROR.
  IF NOT AVAIL B_TT_pdf_param THEN DO:
    CREATE B_TT_pdf_param.
    ASSIGN B_TT_pdf_param.obj_stream    = pdfStream
           B_TT_pdf_param.obj_parameter = "Orientation".
  END.

  IF B_TT_pdf_param.obj_value <> pdfValue THEN DO:
    run ip_pdf_PageWidth(pdfStream,output L_width).
    run ip_pdf_PageHeight(pdfStream,output L_height).

    IF pdfValue = "Landscape" THEN DO:
      RUN pdf_set_PageWidth(pdfStream,L_height).
      RUN pdf_set_PageHeight(pdfStream,L_width).
    END.
    ELSE DO:
      RUN pdf_set_PageWidth(pdfStream,L_width).
      RUN pdf_set_PageHeight(pdfStream,L_height).
    END.
  END.

  B_TT_pdf_param.obj_value = pdfValue.

END. /* pdf_set_Orientation */

PROCEDURE pdf_set_VerticalSpace :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_VerticalSpace","Cannot find Stream!").
    RETURN .
  END.

  IF pdfValue = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_VerticalSpace","Vertical Space cannot be zero!").
    RETURN .
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "VerticalSpace" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "VerticalSpace".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_VerticalSpace */

PROCEDURE pdf_set_LeftMargin :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_LeftMargin","Cannot find Stream!").
    RETURN .
  END.

  IF pdfValue = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_LeftMargin","Left Margin cannot be zero!").
    RETURN .
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "LeftMargin" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "LeftMargin".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_LeftMargin */

PROCEDURE pdf_set_TopMargin :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TopMargin","Cannot find Stream!").
    RETURN .
  END.

  IF pdfValue = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_TopMargin","Top Margin cannot be zero!").
    RETURN .
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "TopMargin" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "TopMargin".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_TopMargin */

PROCEDURE pdf_set_BottomMargin :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_BottomMargin","Cannot find Stream!").
    RETURN .
  END.

  IF pdfValue <= 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_BottomMargin","Bottom Margin cannot be <= zero!").
    RETURN .
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "BottomMargin" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "BottomMargin".
  END.

  TT_pdf_param.obj_value = STRING(pdfValue).

END. /* pdf_set_BottomMargin */

PROCEDURE pdf_set_PaperType :
  DEFINE INPUT PARAMETER pdfStream          AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfValue           AS CHARACTER NO-UNDO.

  def var cpdf_orientation as char no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_PaperType","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_option  AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_width   AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_height  AS INTEGER NO-UNDO.

  L_Option = "A0,A1,A2,A3,A4,A5,A6,B5,LETTER,LEGAL,LEDGER".

  IF LOOKUP(pdfValue,L_option) = 0 THEN DO:
    RUN pdf_error(pdfStream,"pdf_set_PaperType","Invalid Paper Type option passed!").
    RETURN .
  END.

  FIND FIRST TT_pdf_param
       WHERE TT_pdf_param.obj_stream    = pdfStream
         AND TT_pdf_param.obj_parameter = "PaperType" EXCLUSIVE-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_param THEN DO:
    CREATE TT_pdf_param.
    ASSIGN TT_pdf_param.obj_stream    = pdfStream
           TT_pdf_param.obj_parameter = "PaperType".
  END.

  /* Set the Paper Type */
  TT_pdf_param.obj_value = pdfValue.

  /* Determine the Paper Height and Width */
  CASE pdfValue:
    WHEN "A0" THEN
      ASSIGN L_width  = 2380
             L_height = 3368.
    WHEN "A1" THEN
      ASSIGN L_width  = 1684
             L_height = 2380.
    WHEN "A2" THEN
      ASSIGN L_width  = 1190
             L_height = 1684.
    WHEN "A3" THEN
      ASSIGN L_width  = 842
             L_height = 1190.
    WHEN "A4" THEN
      ASSIGN L_width  = 595
             L_height = 842.
    WHEN "A5" THEN
      ASSIGN L_width  = 421
             L_height = 595.
    WHEN "A6" THEN
      ASSIGN L_width  = 297
             L_height = 421.
    WHEN "B5" THEN
      ASSIGN L_width  = 501
             L_height = 709.
    WHEN "LETTER" THEN
      ASSIGN L_width  = 612
             L_height = 792.
    WHEN "LEGAL" THEN
      ASSIGN L_width  = 612
             L_height = 1008.
    WHEN "LEDGER" THEN
      ASSIGN L_width  = 1224
             L_height = 792.
    OTHERWISE
      ASSIGN L_width  = 612
             L_height = 792.
  END CASE.

  run ip_pdf_Orientation(pdfStream,output cpdf_orientation).
  /* Now Set the Page Height and Width Parameters */
  IF cpdf_Orientation = "Portrait" THEN DO:
    RUN pdf_set_PageWidth(pdfStream,L_width).
    RUN pdf_set_PageHeight(pdfStream,L_height).
  END.
  ELSE DO:
    RUN pdf_set_PageWidth(pdfStream,L_height).
    RUN pdf_set_PageHeight(pdfStream,L_width).
  END.

END. /* pdf_set_PaperType */

/* ---------------------- Define INTERNAL PROCEDURES ----------------------- */
PROCEDURE pdf_init_param:
  DEFINE INPUT PARAMETER pdfStream AS CHARACTER NO-UNDO.

  def var iPageHeight as int no-undo.
  def var iTopMargin  as int no-undo.

  /* Create Parameters */
  RUN pdf_set_Orientation(pdfStream,"Portrait").
  RUN pdf_set_PaperType(pdfStream,"LETTER").  /* This also default the PageWidth
                                             and PageHeight Parameters */
  RUN pdf_set_Font(pdfStream,"Courier",10.0). /* This also sets the PointSize */
  RUN pdf_set_VerticalSpace(pdfStream,10).
  RUN pdf_set_LeftMargin(pdfStream,10).
  RUN pdf_set_TopMargin(pdfStream,50).
  RUN pdf_set_BottomMargin(pdfStream,1).
  RUN pdf_set_Angle(pdfStream,0).
  run ip_pdf_PageHeight(pdfstream, output iPageHeight).
  run ip_pdf_TopMargin(pdfstream, output iTopMargin).
  RUN pdf_set_TextY(pdfStream,(iPageHeight - iTopMargin)).
  RUN pdf_set_TextX(pdfStream,0).
  RUN pdf_set_GraphicY(pdfStream,0).
  RUN pdf_set_GraphicX(pdfStream,0).
  RUN pdf_set_TextRed(pdfStream,.0).
  RUN pdf_set_TextGreen(pdfStream,.0).
  RUN pdf_set_TextBlue(pdfStream,.0).

END. /* pdf_init_param */

PROCEDURE pdf_Header :
  DEFINE INPUT PARAMETER P_Stream AS CHARACTER NO-UNDO.

  def var cAuthor   as char no-undo.
  def var cProducer as char no-undo.
  def var cCreator  as char no-undo.
  def var cSubject  as char no-undo.
  def var cTitle    as char no-undo.
  def var cKeywords as char no-undo.


  /* Version Compatibilities */
  PUT STREAM S_pdf_inc UNFORMATTED
      "%PDF-1.4" SKIP.

  /* Output 4 Binary Characters (greater than ASCII 128) to indicate to a binary
     file -- randomly selected codes */
  PUT STREAM S_pdf_inc UNFORMATTED
      "%" CHR(244) CHR(244) CHR(244) CHR(244) SKIP.

  /* Display Creation, Title, Producer etc Information */
  run ip_ObjectSequence( p_Stream,1, "Info", output pdf_inc_ObjectSequence ).
  run ip_pdf_get_info(P_Stream,"Author", output cAuthor).
  run ip_pdf_get_info(P_Stream,"Producer", output cProducer).
  run ip_pdf_get_info(P_Stream,"Creator", output cCreator).
  run ip_pdf_get_info(P_Stream,"Subject", output cSubject).
  run ip_pdf_get_info(P_Stream,"Title", output cTitle).
  run ip_pdf_get_info(P_Stream,"Keywords", output cKeywords).
  PUT STREAM S_pdf_inc UNFORMATTED
      /* pdf_inc_ObjectSequence */ "1 0 obj" SKIP
      "<<" SKIP
      "/Author (" cAuthor ")" SKIP
      "/CreationDate (D:" TRIM(STRING(YEAR(TODAY),"9999"))
                          TRIM(STRING(MONTH(TODAY),"99"))
                          TRIM(STRING(DAY(TODAY),"99"))
                          REPLACE(STRING(TIME,"hh:mm:ss"),":","")
                          "-0800)"
                          SKIP
      "/Producer (" cProducer ")" SKIP
      "/Creator (" cCreator ")" SKIP
      "/Subject (" cSubject ")" SKIP
      "/Title (" cTitle ")" SKIP
      "/Keywords (" cKeywords ")" SKIP
      ">>" SKIP
      "endobj" SKIP.

END. /* pdf_header */

PROCEDURE pdf_LoadBase14:

  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  /* ---- Beginning of Courier Fonts ---- */

  /* ObjectSequence(pdfStream, 5, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Courier"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 5
         TT_pdf_font.font_tag   = "/F1"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "FIXED"
         TT_pdf_font.font_width = "600".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Courier-Oblique"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 6
         TT_pdf_font.font_tag   = "/F2"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "FIXED"
         TT_pdf_font.font_width = "600".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Courier-Bold"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 7
         TT_pdf_font.font_tag   = "/F3"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "FIXED"
         TT_pdf_font.font_width = "600".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Courier-BoldOblique"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 8
         TT_pdf_font.font_tag   = "/F4"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "FIXED"
         TT_pdf_font.font_width = "600".

  /* ---- End of Courier Fonts ---- */

  /* ---- Beginning of Helvetica Fonts ---- */

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Helvetica"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 9
         TT_pdf_font.font_tag   = "/F5"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width = FILL("788 ", 31) + "278 278 355 556 556 889 "
                                + "667 222 333 333 389 584 278 333 278 278 556 "
                                + "556 556 556 556 556 556 556 556 556 278 278 "
                                + "584 584 584 556 1015 667 667 722 722 667 611 778 "
                                + "722 278 500 667 556 833 722 778 667 778 722 "
                                + "667 611 722 667 944 667 667 611 278 278 278 "
                                + "469 556 222 556 556 500 556 556 278 556 556 "
                                + "222 222 500 222 833 556 556 556 556 333 500 "
                                + "278 556 500 722 500 500 500 334 260 334 584 "
                                + "333 556 556 167 556 556 556 556 191 333 556 "
                                + "333 333 500 500 556 556 556 278 537 350 222 "
                                + "333 333 556 1000 1000 611 333 333 333 333 333 "
                                + "333 333 333 333 333 333 333 333 1000 1000 370 "
                                + "556 778 1000 365 889 278 222 611 944 611".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Helvetica-Oblique"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 10
         TT_pdf_font.font_tag   = "/F6"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width = FILL("788 ", 31) + "278 278 355 556 556 889 "
                                + "667 222 333 333 389 584 278 333 278 278 556 "
                                + "556 556 556 556 556 556 556 556 556 278 278 "
                                + "584 584 584 556 1015 667 667 722 722 667 611 "
                                + "778 722 278 500 667 556 833 722 778 667 778 "
                                + "722 667 611 722 667 944 667 667 611 278 278 "
                                + "278 469 556 222 556 556 500 556 556 278 556 "
                                + "556 222 222 500 222 833 556 556 556 556 333 "
                                + "500 278 556 500 722 500 500 500 334 260 334 "
                                + "584 333 556 556 167 556 556 556 556 191 333 "
                                + "556 333 333 500 500 556 556 556 278 537 350 "
                                + "222 333 333 556 1000 1000 611 333 333 333 333 "
                                + "333 333 333 333 333 333 333 333 333 1000 "
                                + "1000 370 556 778 1000 365 889 278 222 611 "
                                + "944 611".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Helvetica-Bold"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 11
         TT_pdf_font.font_tag   = "/F7"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width = FILL("788 ", 31) + "278 333 474 556 556 889 "
                                + "722 278 333 333 389 584 278 333 278 278 556 "
                                + "556 556 556 556 556 556 556 556 556 333 333 "
                                + "584 584 584 611 975 722 722 722 722 667 611 "
                                + "778 722 278 556 722 611 833 722 778 667 778 "
                                + "722 667 611 722 667 944 667 667 611 333 278 "
                                + "333 584 556 278 556 611 556 611 556 333 611 "
                                + "611 278 278 556 278 889 611 611 611 611 389 "
                                + "556 333 611 556 778 556 556 500 389 280 389 "
                                + "584 333 556 556 167 556 556 556 556 238 500 "
                                + "556 333 333 611 611 556 556 556 278 556 350 "
                                + "278 500 500 556 1000 1000 611 333 333 333 333 "
                                + "333 333 333 333 333 333 333 333 333 1000 "
                                + "1000 370 611 778 1000 365 889 278 278 611 "
                                + "944 611".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Helvetica-BoldOblique"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 12
         TT_pdf_font.font_tag   = "/F8"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width = FILL("788 ", 31) + "278 333 474 556 556 889 "
                                + "722 278 333 333 389 584 278 333 278 278 556 "
                                + "556 556 556 556 556 556 556 556 556 333 333 "
                                + "584 584 584 611 975 722 722 722 722 667 611 "
                                + "778 722 278 556 722 611 833 722 778 667 778 "
                                + "722 667 611 722 667 944 667 667 611 333 278 "
                                + "333 584 556 278 556 611 556 611 556 333 611 "
                                + "611 278 278 556 278 889 611 611 611 611 389 "
                                + "556 333 611 556 778 556 556 500 389 280 389 "
                                + "584 333 556 556 167 556 556 556 556 238 500 "
                                + "556 333 333 611 611 556 556 556 278 556 350 "
                                + "278 500 500 556 1000 1000 611 333 333 333 "
                                + "333 333 333 333 333 333 333 333 333 333 "
                                + "1000 1000 370 611 778 1000 365 889 278 278 "
                                + "611 944 611".

  /* ---- End of Helvetica Fonts ---- */

  /* ---- Beginning of Times Roman Fonts ---- */

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Times-Roman"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 13
         TT_pdf_font.font_tag   = "/F9"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width =   FILL("788 ", 31) + "250 " + "333 " + "408 "
                                + "500 500 833 778 180 333 333 500 564 250 "
                                + "333 250 278 500 500 500 500 500 500 500 "
                                + "500 500 500 278 278 564 564 564 444 921 "
                                + "722 667 667 722 611 556 722 722 333 389 "
                                + "722 611 889 722 722 556 722 667 556 611 "
                                + "722 722 944 722 722 611 333 278 333 469 "
                                + "500 333 444 500 444 500 444 333 500 500 "
                                + "278 278 500 278 778 500 500 500 500 333 "
                                + "389 278 500 500 722 500 500 444 480 200 "
                                + "480 541 778 500 578 333 500 444 1000 500 "
                                + "500 333 1000 556 333 889 667 611 722 444 "
                                + "333 333 444 444 350 500 1000 333 980 389 "
                                + "333 722 486 444 722 250 333 500 500 500 "
                                + "500 200 500 333 760 276 500 564 333 760 "
                                + "500 400 549 300 300 333 576 453 250 333 "
                                + "300 310 500 750 750 750 444 722 722 722 "
                                + "722 722 722 889 667 611 611 611 611 333 "
                                + "333 333 333 722 722 722 722 722 722 722 "
                                + "564 722 722 722 722 722 722 556 500 444 "
                                + "444 444 444 444 444 667 444 444 444 444 "
                                + "444 278 278 278 278 500 500 500 500 500 "
                                + "500 500 549 500 500 500 500 500 500 500 "
                                + "500".


  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Times-Italic"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 14
         TT_pdf_font.font_tag   = "/F10"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width =   FILL("788 ", 31) + "250 333 420 500 500 "
                                + "833 778 333 333 333 500 675 250 333 250 278 "
                                + "500 500 500 500 500 500 500 500 500 500 333 "
                                + "333 675 675 675 500 920 611 611 667 722 611 "
                                + "611 722 722 333 444 667 556 833 667 722 611 "
                                + "500 559 722 611 833 611 556 556 389 278 389 "
                                + "422 500 333 500 500 444 500 444 278 500 500 "
                                + "278 278 444 278 722 500 500 500 500 389 389 "
                                + "278 500 444 667 444 444 389 400 278 400 541 "
                                + "389 500 500 167 500 500 500 500 214 556 500 "
                                + "333 333 500 500 500 500 500 250 523 350 333 "
                                + "556 556 500 889 1000 500 333 333 333 333 333 "
                                + "333 333 333 333 333 333 333 333 889 889 276"
                                + "556 722 944 310 667 278 278 500 667 500".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Times-Bold"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 15
         TT_pdf_font.font_tag   = "/F11"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width =  FILL("788 ", 31) + "250 333 555 500 500 "
                                + "1000 833 333 333 333 500 570 250 333 250 "
                                + "278 500 500 500 500 500 500 500 500 500 500 "
                                + "333 333 570 570 570 500 930 722 667 722 722 "
                                + "667 611 778 778 389 500 778 667 944 722 778 "
                                + "611 778 722 556 667 722 722 1000 722 722 667 "
                                + "333 278 333 581 500 333 500 556 444 556 444 "
                                + "333 500 556 278 333 556 278 833 556 500 556 "
                                + "556 444 389 333 556 500 722 500 500 444 394 "
                                + "220 394 520 333 500 500 167 500 500 500 500 "
                                + "278 500 500 333 333 556 556 500 500 500 250 "
                                + "540 350 333 500 500 500 1000 1000 500 333 333 "
                                + "333 333 333 333 333 333 333 333 333 333 333 "
                                + "1000 1000 300 667 778 1000 330 722 278 278 "
                                + "500 722 556".

  /* ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font"). */
  /* Create Associated Object */
  CREATE TT_pdf_font.
  ASSIGN TT_pdf_font.font_name  = "Times-BoldItalic"
         TT_pdf_font.font_file  = "PDFBASE14"
         TT_pdf_font.font_afm   = ""
         TT_pdf_font.font_obj   = 16
         TT_pdf_font.font_tag   = "/F12"
         TT_pdf_font.obj_stream = pdfStream
         TT_pdf_font.font_type  = "VARIABLE"
         TT_pdf_font.font_width = FILL("788 ", 31) + "250 389 555 500 500 833 "
                                + "778 333 333 333 500 570 250 333 250 278 500 "
                                + "500 500 500 500 500 500 500 500 500 333 333 "
                                + "570 570 570 500 832 667 667 667 722 667 667 "
                                + "722 778 389 500 667 611 889 722 722 611 722 "
                                + "667 556 611 722 667 889 667 611 611 333 278 "
                                + "333 570 500 333 500 500 444 500 444 333 500 "
                                + "556 278 278 500 278 778 556 500 500 500 389 "
                                + "278 556 444 667 500 444 389 348 220 348 570 "
                                + "389 500 500 167 500 500 500 500 278 500 500 "
                                + "333 333 556 556 500 500 500 250 500 350 333 "
                                + "500 500 500 1000 1000 500 333 333 333 333 333 "
                                + "333 333 333 333 333 333 333 333 1000 944 266 "
                                + "611 722 944 300 722 278 278 500 722 500".

  /* ---- End of Times Roman Fonts ---- */

  pdf_inc_ObjectSequence = 16.

END. /* pdf_LoadBase14 */

PROCEDURE pdf_Encoding:
  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  run ip_ObjectSequence(pdfStream,4, "Encoding", output pdf_inc_ObjectSequence).
  PUT STREAM S_pdf_inc UNFORMATTED
      /* pdf_inc_ObjectSequence */ "4 0 obj" SKIP
      "<<" SKIP
      "/Type /Encoding" SKIP
      "/BaseEncoding /WinAnsiEncoding" SKIP
      ">>" SKIP
      "endobj" SKIP.

END. /* pdf_Encoding */

PROCEDURE pdf_Resources:
  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_fontobj AS CHARACTER NO-UNDO.

  FOR EACH TT_pdf_font NO-LOCK:
    ASSIGN L_fontobj = L_fontobj + " " + TT_pdf_font.font_tag + " "
                     + STRING(TT_pdf_font.font_obj) + " 0 R".
  END.

  Vuse-font = "/F1".

  run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Resource",
                        output pdf_inc_ObjectSequence).
  pdf-Res-Object = pdf_inc_ObjectSequence.
  PUT STREAM S_pdf_inc UNFORMATTED
      pdf_inc_ObjectSequence " 0 obj" SKIP
      "<<" SKIP
      "  /Font << " L_fontobj " >>" SKIP
      "  /ProcSet [ /PDF /Text /ImageC ]" SKIP
      "  /XObject << " .

  /* Output Image Definitions */
  FOR EACH TT_pdf_image:
    PUT STREAM S_pdf_inc UNFORMATTED
        TT_pdf_image.image_tag " " TT_pdf_image.image_obj " 0 R ".
  END.

  PUT STREAM S_pdf_inc UNFORMATTED
      " >>" SKIP
      ">>" SKIP
      "endobj" SKIP.

END. /* pdf_Resources */

PROCEDURE pdf_Content:
  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_font    AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_Loop      AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_Page      AS INTEGER NO-UNDO.
  def    var      iPage       as int     no-undo.
  def    var      iPage1      as int     no-undo.
  def    var      iPageHeight as int     no-undo.
  def    var      iTopMargin  as int     no-undo.
  def    var      iLeftMargin as int     no-undo.

  run ip_pdf_page(pdfstream, output iPage).

  /* Produce each Page one at a time */
  DO L_Loop = 1 TO iPage:
    IF CAN-FIND(FIRST TT_pdf_content WHERE TT_pdf_content.obj_page = L_Loop)
    THEN DO:

      L_page = L_page + 1.
      /* Start Page Definition */
      RUN pdf_Definition (pdfStream ,L_page, IF L_page = 1 THEN FALSE ELSE TRUE).
      run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Content",
                            output pdf_inc_ObjectSequence).
      PUT STREAM S_pdf_inc UNFORMATTED
          pdf_inc_ObjectSequence " 0 obj" SKIP
          "<<" SKIP
          "/Length " (pdf_inc_ObjectSequence + 1) " 0 R" SKIP
          ">>" SKIP
          "stream" SKIP.
      pdf-Stream-Start = SEEK(S_pdf_inc).

      /* Process Watermark Content */
      IF CAN-FIND(FIRST TT_pdf_content
                  WHERE TT_pdf_content.obj_stream  = pdfStream
                    AND TT_pdf_content.obj_page    = L_page
                    AND TT_pdf_content.obj_type    = "WATERMARK" NO-LOCK)
      THEN DO:
        FOR EACH TT_pdf_content WHERE TT_pdf_content.obj_stream  = pdfStream
                                  AND TT_pdf_content.obj_page    = L_page
                                  AND TT_pdf_content.obj_type    = "WATERMARK"
                                  NO-LOCK
                                  BREAK BY TT_pdf_content.obj_seq:
          PUT STREAM S_pdf_inc UNFORMATTED
             TT_pdf_content.obj_content CHR(13).
        END.
      END. /* Cand find Watermark Content */

      /* Process Rectangle Content */
      IF CAN-FIND(FIRST TT_pdf_content
                  WHERE TT_pdf_content.obj_stream  = pdfStream
                    AND TT_pdf_content.obj_page    = L_page
                    AND TT_pdf_content.obj_type BEGINS "GRAPHIC" NO-LOCK)
      THEN DO:
        PUT STREAM S_pdf_inc UNFORMATTED
            "q" CHR(13).
        FOR EACH TT_pdf_content WHERE TT_pdf_content.obj_stream      = pdfStream
                                  AND TT_pdf_content.obj_page        = L_page
                                  AND TT_pdf_content.obj_type BEGINS "GRAPHIC"
                                  NO-LOCK
                                  BREAK BY TT_pdf_content.obj_seq:
          PUT STREAM S_pdf_inc UNFORMATTED
             TT_pdf_content.obj_content CHR(13).
        END.
        PUT STREAM S_pdf_inc UNFORMATTED
            "Q" CHR(13).
      END. /* Cand find Graphic Content */

      /* Process Image Content */
      FOR EACH TT_pdf_content WHERE TT_pdf_content.obj_stream      = pdfStream
                                AND TT_pdf_content.obj_page        = L_page
                                AND TT_pdf_content.obj_type BEGINS "IMAGE"
                                NO-LOCK
                                BREAK BY TT_pdf_content.obj_seq:
        PUT STREAM S_pdf_inc UNFORMATTED
           "q " TT_pdf_content.obj_content " Q" CHR(13).
      END.

      /* Process Text (at, inline, to) Content */
      run ip_pdf_PageHeight(pdfStream, output iPageHeight).
      run ip_pdf_TopMargin(pdfStream, output iTopMargin).
      run ip_pdf_LeftMargin(pdfStream, output iLeftMargin).
      PUT STREAM S_pdf_inc UNFORMATTED
          "BT" CHR(13)
          Vuse-font " 10 Tf" CHR(13)
          "1 0 0 1 " iLeftMargin " " (iPageHeight - iTopMargin)
          " Tm" CHR(13)
          "10 TL" CHR(13).

      FOR EACH TT_pdf_content WHERE TT_pdf_content.obj_stream  = pdfStream
                                AND TT_pdf_content.obj_page    = L_page
                                AND TT_pdf_content.obj_type BEGINS "TEXT"
                                /* AND TT_pdf_content.obj_type <> "TEXTXY" */ NO-LOCK
          BREAK BY TT_pdf_content.obj_line DESC
                BY TT_pdf_content.obj_seq:

        IF INDEX(TT_pdf_content.obj_content,"@@TotalPages-" + pdfStream) > 0 THEN
        do:
          run ip_pdf_page(pdfstream, output iPage1).
          TT_pdf_content.obj_content = REPLACE(TT_pdf_content.obj_content,
                                               "@@TotalPages-" + pdfStream,
                                               STRING(iPage1)).
        end.

        PUT STREAM S_pdf_inc UNFORMATTED
           TT_pdf_content.obj_content CHR(13).

      END. /* each text object on page */

      PUT STREAM S_pdf_inc UNFORMATTED
          "ET" CHR(13).

      /* End Page Definition */
      pdf-Stream-End = SEEK(S_pdf_inc).
      PUT STREAM S_pdf_inc UNFORMATTED
          "endstream" SKIP
          "endobj" SKIP.

      /* Output Length */
      RUN pdf_length (pdfStream, pdf-Stream-End - pdf-Stream-Start).

    END. /* Can-find page content */

  END. /* loop for each page */

  /* This will set the PDF Page to the max actual Page number */
  RUN pdf_set_Page(pdfStream, L_page).

  RUN pdf_catalog (pdfStream).
  RUN pdf_ListPages (pdfStream).
END. /* pdf_Content */

PROCEDURE pdf_definition:
  DEFINE INPUT PARAMETER pdfStream      AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER P_page         AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER P_incl-annot   AS LOGICAL NO-UNDO.

  run ip_ObjectSequence (pdfStream, pdf_inc_ObjectSequence + 1, "Definition",
                         output pdf_inc_ObjectSequence).
  PUT STREAM S_pdf_inc UNFORMATTED
      pdf_inc_ObjectSequence " 0 obj" SKIP
      "<<" SKIP
      "/Type /Page" SKIP
      "/Parent 3 0 R" SKIP
      "/Resources " pdf-Res-Object " 0 R" SKIP
      "/Contents " (pdf_inc_ObjectSequence + 1) " 0 R" SKIP
      "/Rotate 0" SKIP
      "/Annots [ ".


      /* Output Link Definitions */
  FOR EACH TT_pdf_link WHERE TT_pdf_link.obj_stream = pdfStream
                         AND TT_pdf_link.link_page  = p_Page:
    PUT STREAM S_pdf_inc UNFORMATTED
        TT_pdf_link.link_obj " 0 R" SKIP.
  END.


  PUT STREAM S_pdf_inc UNFORMATTED
      " ]" SKIP
      ">>" SKIP
      "endobj" SKIP.

END. /* pdf_definition */

PROCEDURE pdf_Length:

  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER P_length   AS INTEGER NO-UNDO.

  run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Length",
                        output pdf_inc_ObjectSequence).
  PUT STREAM S_pdf_inc UNFORMATTED
      pdf_inc_ObjectSequence " 0 obj" SKIP
      P_length SKIP
      "endobj" SKIP.

END. /* pdf_Length */

PROCEDURE pdf_Catalog :
  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  run ip_ObjectSequence(pdfStream, 2, "Catalog", output pdf_inc_ObjectSequence).
  PUT STREAM S_pdf_inc UNFORMATTED
      "2 0 obj" SKIP
      "<<" SKIP
      "/Type /Catalog" SKIP
      "/Pages 3 0 R" SKIP
      /* "/PageLayout /OneColumn" SKIP */
      ">>" SKIP
      "endobj" SKIP.

END. /* pdf_Catalog */

PROCEDURE pdf_ListPages :

  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  def var iPage       as int no-undo.
  def var iPageWidth  as int no-undo.
  def var iPageHeight as int no-undo.

  run ip_ObjectSequence(pdfStream, 3, "Pages", output pdf_inc_ObjectSequence).
  run ip_pdf_page(pdfstream, output iPage).
  run ip_pdf_PageWidth(pdfStream,output iPageWidth).
  run ip_pdf_PageHeight(pdfStream,output iPageheight).
  PUT STREAM S_pdf_inc UNFORMATTED
      "3 0 obj" SKIP
      "<<" SKIP
      "/Type /Pages" SKIP
      "/Count " iPage SKIP
      "/MediaBox [ 0 0 " iPageWidth " " iPageHeight " ]" SKIP
      "/Kids [ ".

  FOR EACH TT_pdf_object WHERE TT_pdf_object.obj_stream = pdfStream
                           AND TT_pdf_object.obj_desc = "Content" NO-LOCK:
    PUT STREAM S_pdf_inc UNFORMATTED
        (TT_pdf_object.obj_nbr - 1) " 0 R ".
  END. /* Display Pages */

  PUT STREAM S_pdf_inc UNFORMATTED
      "]" SKIP
      ">>" SKIP
      "endobj" SKIP.

END. /* pdf_ListPages */

PROCEDURE pdf_xref :

  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_ctr AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_obj AS INTEGER NO-UNDO.

  run ip_ObjectSequence(pdfStream, 0, "Xref", output pdf_inc_ObjectSequence).
  FOR EACH TT_pdf_object WHERE TT_pdf_object.obj_stream = pdfStream:
    L_ctr = L_ctr + 1.
  END.

  /* Get the Xref start point */
  pdf-Stream-Start = SEEK(S_pdf_inc).

  PUT STREAM S_pdf_inc UNFORMATTED
      "xref" SKIP
      "0 " L_ctr SKIP.

  FOR EACH TT_pdf_object BREAK BY TT_pdf_object.obj_nbr:
    IF FIRST( TT_pdf_object.obj_nbr) THEN
      PUT STREAM S_pdf_inc CONTROL
          "0000000000"
          " "
          TT_pdf_object.gen_nbr " "
          TT_pdf_object.obj_type CHR(13) CHR(10).
    ELSE DO:
      TT_pdf_object.obj_off = TT_pdf_object.obj_off - 1.
      PUT STREAM S_pdf_inc CONTROL
          STRING(TT_pdf_object.obj_off, "9999999999")
          " "
          STRING(TT_pdf_object.gen_nbr, "99999") " "
          TT_pdf_object.obj_type CHR(13) CHR(10).
    END.
  END. /* each Object */

  FIND LAST TT_pdf_object NO-LOCK NO-ERROR.

  PUT STREAM S_pdf_inc UNFORMATTED
      "trailer" SKIP
      "<<" SKIP
      "/Size " (TT_pdf_object.obj_nbr + 1) SKIP
      "/Root 2 0 R" SKIP
      "/Info 1 0 R" SKIP
      ">>" SKIP
      "startxref" SKIP
      pdf-Stream-Start SKIP
      "%%EOF" SKIP.

END. /* pdf_xref */

PROCEDURE pdf_Fonts:

  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  FOR EACH TT_pdf_font WHERE TT_pdf_font.font_file = "PDFBASE14"
      BY TT_pdf_font.font_obj:

    run ip_ObjectSequence(pdfStream, TT_pdf_font.font_obj, "Font",
                          output pdf_inc_ObjectSequence).
    PUT STREAM S_pdf_inc UNFORMATTED
          TT_pdf_font.FONT_obj " 0 obj" SKIP
          "<<" SKIP
          "/Type /Font" SKIP
          "/Subtype /Type1" SKIP
          "/Name " TT_pdf_font.font_tag SKIP
          "/Encoding 4 0 R" SKIP
          "/BaseFont /" TT_pdf_font.font_name SKIP
          ">>" SKIP
          "endobj" SKIP.
  END.

END. /* pdf_Fonts */

PROCEDURE pdf_Load_fonts :
  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_data  AS RAW NO-UNDO.
  DEFINE VARIABLE L_start AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_end   AS INTEGER NO-UNDO.

  FOR EACH TT_pdf_font WHERE TT_pdf_font.obj_stream = pdfStream
                         AND TT_pdf_font.font_file <> "PDFBASE14":

    /*
    RUN pdf_ParseAFMFile
        (OUTPUT L_afm_ItalicAngle,
         OUTPUT L_afm_Ascender,
         OUTPUT L_afm_Descender,
         OUTPUT L_afm_FontBBox,
         OUTPUT L_afm_FirstChar,
         OUTPUT L_afm_LastChar,
         OUTPUT L_afm_Widths,
         OUTPUT L_afm_IsFixedPitch,
         OUTPUT L_afm_Flags) NO-ERROR.

    /* If any errors occur while parsing AFM file then ignore the font */
    IF ERROR-STATUS:ERROR THEN NEXT.
    **/

    FILE-INFO:FILE-NAME = TT_pdf_font.font_file.

    /* igc - Added Sept 10, 2002 */
    run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1,
                          "FontDescriptor", output pdf_inc_ObjectSequence).
    TT_pdf_font.font_descr  = pdf_inc_ObjectSequence.

    /* Output the Font Descriptor */
    PUT STREAM S_pdf_inc UNFORMATTED
        TT_pdf_font.font_descr " 0 obj" SKIP
        "<< /Type /FontDescriptor" SKIP
        "   /Ascent " TT_pdf_font.afm_Ascender SKIP
        "   /Descent " TT_pdf_font.afm_Descender SKIP
        "   /Flags " TT_pdf_font.afm_Flags SKIP
        "   /FontBBox [ " TT_pdf_font.afm_FontBBox " ]" SKIP
        "   /FontName /" TT_pdf_font.font_name SKIP
        "   /ItalicAngle " TT_pdf_font.afm_ItalicAngle  SKIP
        "   /FontFile2 " (TT_pdf_font.font_descr + 2) " 0 R" SKIP
        ">>" SKIP
        "endobj" SKIP.

    /* igc - Added Sept 10, 2002 */
    run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Font",
                          output pdf_inc_ObjectSequence).
    TT_pdf_font.font_obj    = pdf_inc_ObjectSequence.

    PUT STREAM S_pdf_inc UNFORMATTED
      TT_pdf_font.font_obj " 0 obj" SKIP
      "<<" SKIP
      "/Type /Font" SKIP
      "/Subtype /TrueType" SKIP
      "/FirstChar " TT_pdf_font.afm_FirstChar SKIP
      "/LastChar " TT_pdf_font.afm_LastChar SKIP
      "/Widths [ " TT_pdf_font.afm_widths " ]" SKIP
      "/Encoding /WinAnsiEncoding" SKIP
      "/BaseFont /" TT_pdf_font.font_name SKIP
      "/FontDescriptor " TT_pdf_font.font_descr " 0 R" SKIP
      ">>" SKIP
      "endobj" SKIP.

    /*** igc - Nov 4, 2002 - removed. Now uses OS-APPEND
    /* Determine the File Size -- cannot use FILE-INFO:FILE-SIZE since that is
       V9 specific code */
    INPUT STREAM S_pdf_inp FROM VALUE(TT_pdf_font.font_file) BINARY NO-MAP
      NO-CONVERT UNBUFFERED.
      SEEK STREAM S_pdf_inp TO END.
      LENGTH(L_data) = IF SEEK(S_pdf_inp) > 16832 THEN 16832 ELSE SEEK(S_pdf_inp).
    INPUT STREAM S_pdf_inp CLOSE.
    ****/

    /* igc - Added Sept 10, 2002 */
    run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "FontStream",
                          output pdf_inc_ObjectSequence).
    TT_pdf_font.font_stream = pdf_inc_ObjectSequence.

    /* Display Embedded Font Stream */
    PUT STREAM S_pdf_inc UNFORMATTED
        TT_pdf_font.font_stream " 0 obj" SKIP
        "<< /Length " (TT_pdf_font.font_stream + 1) " 0 R /Length1 "
        (TT_pdf_font.font_stream + 1) " 0 R >>" SKIP
        "stream" SKIP.

    /* Get PDF Stream Start Offset */
    L_start = SEEK(S_pdf_inc).

    /*** igc - Nov 4, 2002 - removed. Now Uses OS-APPEND
    INPUT STREAM S_pdf_inp FROM VALUE(TT_pdf_font.font_file) BINARY NO-MAP
      NO-CONVERT UNBUFFERED.
      REPEAT:
        IMPORT STREAM S_pdf_inp UNFORMATTED L_data.
        PUT STREAM S_pdf_inc CONTROL L_data.
      END.
    INPUT STREAM S_pdf_inp CLOSE.
    ****/

    OUTPUT STREAM S_pdf_inc CLOSE.
    OS-APPEND VALUE(FILE-INFO:FILE-NAME) VALUE(TT_pdf_stream.obj_file ).
    OUTPUT STREAM S_pdf_inc TO VALUE( TT_pdf_stream.obj_file ) UNBUFFERED APPEND.

    /* Get PDF Stream End Offset */
    L_end = SEEK(S_pdf_inc).

    PUT STREAM S_pdf_inc UNFORMATTED
        SKIP SKIP "endstream" SKIP
        "endobj" SKIP.

    /* igc - Added Sept 10, 2002 */
    run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "FontLength",
                          output pdf_inc_ObjectSequence).
    TT_pdf_font.font_len = pdf_inc_ObjectSequence.

    /* Put out Length */
    PUT STREAM S_pdf_inc UNFORMATTED
        TT_pdf_font.font_len " 0 obj" SKIP
        "  " (L_end - L_start) SKIP
        "endobj" SKIP.

  END. /* each TTfont */

END. /* pdf_Load_fonts */

PROCEDURE pdf_ParseAFMFile:
  DEFINE OUTPUT PARAMETER P_afm_ItalicAngle  AS INTEGER NO-UNDO.
  DEFINE OUTPUT PARAMETER P_afm_Ascender     AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER P_afm_Descender    AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER P_afm_FontBBox     AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER P_afm_FirstChar    AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER P_afm_LastChar     AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER P_afm_Widths       AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER P_afm_IsFixedPitch AS CHARACTER NO-UNDO INIT "0".
  DEFINE OUTPUT PARAMETER P_afm_flags        AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_data  AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_key   AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_flag  AS CHARACTER NO-UNDO
         INIT "00000000000000000000000000100010".
  /* Bit 6 (above) is set to identify NonSymbolic Fonts -- or Fonts that use
     the Standard Latin Character Set */

  DEFINE VARIABLE L_int   AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_Loop  AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_exp   AS INTEGER NO-UNDO.

  ASSIGN P_afm_ItalicAngle  = 0
         P_afm_Descender    = ""
         P_afm_FontBBox     = ""
         P_afm_IsFixedPitch = ""
         P_afm_FirstChar    = ""
         P_afm_LastChar     = ""
         P_afm_Widths       = "".

  INPUT STREAM S_pdf_inp FROM VALUE(TT_pdf_font.font_afm) BINARY NO-CONVERT NO-MAP NO-ECHO.

    REPEAT:
      IMPORT STREAM S_pdf_inp UNFORMATTED L_data.
      L_Key = ENTRY(1, L_data, " ") NO-ERROR.
      IF ERROR-STATUS:ERROR THEN NEXT.

      CASE L_key:
        WHEN "ItalicAngle" THEN
          P_afm_ItalicAngle = INT(ENTRY( 2, L_data, " ")) NO-ERROR.

        WHEN "Ascender" THEN
          P_afm_Ascender = ENTRY( 2, L_data, " ") NO-ERROR.

        WHEN "Descender" THEN
          P_afm_Descender = ENTRY( 2, L_data, " ") .

        WHEN "FontBBox" THEN
          ASSIGN P_afm_FontBBox = REPLACE(L_data,"FontBBox ","").

        WHEN "IsFixedPitch" THEN
          P_afm_IsFixedPitch = IF ENTRY(2,L_data, " ") = "True" THEN "1" ELSE "0".

        WHEN "C" THEN DO:
          IF P_afm_FirstChar = "" THEN
            P_afm_FirstChar = ENTRY(2, L_data, " ") NO-ERROR.

          ASSIGN P_afm_Widths = P_afm_widths + " "
                              + ENTRY(5, L_data, " ") NO-ERROR.

          P_afm_LastChar = ENTRY(2, L_data, " ") NO-ERROR.
        END.
      END CASE.
    END. /* REPEAT */

  INPUT STREAM S_pdf_inp CLOSE.

  /* Determine Font Flags */
  IF P_afm_IsFixedPitch = "0" THEN
    OVERLAY( L_Flag, 32, 1, "CHARACTER") = "1".

  DO L_loop = LENGTH(L_Flag) TO 1 BY -1 :
    IF SUBSTR(L_flag,L_loop,1) = "1" THEN
      L_int = L_int + EXP(2, L_exp).

    L_exp = L_exp + 1.
  END.

  P_afm_Flags = STRING( L_int ).

END. /* pdf_ParseAFMFile */

PROCEDURE pdf_load_images:

  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_data  AS RAW NO-UNDO.
  DEFINE VARIABLE L_start AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_end   AS INTEGER NO-UNDO.

  FOR EACH TT_pdf_image WHERE TT_pdf_image.obj_stream = pdfStream:

    FILE-INFO:FILE-NAME = TT_pdf_image.image_file.

    /* igc - Added Sept 10, 2002 */
    run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Image",
                          output pdf_inc_ObjectSequence).
    TT_pdf_image.image_obj = pdf_inc_ObjectSequence.

    PUT STREAM S_pdf_inc UNFORMATTED
        TT_pdf_image.image_obj " 0 obj" SKIP
        "<<" SKIP
        "/Type /XObject" SKIP
        "/Subtype /Image" SKIP
        "/Name " TT_pdf_image.image_tag SKIP
        "/Width " TT_pdf_image.image_w SKIP
        "/Height " TT_pdf_image.image_h SKIP
        "/BitsPerComponent 8" SKIP
        "/ColorSpace /DeviceRGB" SKIP
        "/Length " (TT_pdf_image.image_obj + 1) " 0 R" SKIP
        "/Filter /DCTDecode" SKIP
        ">>" SKIP
        "stream" SKIP.

      /** igc - Nov 4, 2002 - removed now use OS-APPEND
      /* Determine the File Size -- cannot use FILE-INFO:FILE-SIZE since that is
         V9 specific code */
      INPUT STREAM S_pdf_inp FROM VALUE(TT_pdf_image.image_file) BINARY NO-MAP NO-CONVERT UNBUFFERED.
        SEEK STREAM S_pdf_inp TO END.
        LENGTH(L_data) = IF SEEK(S_pdf_inp) > 16832 THEN 16832 ELSE SEEK(S_pdf_inp).
      INPUT STREAM S_pdf_inp CLOSE.
      ***/

      /* Get PDF Stream Start Offset */
      L_start = SEEK(S_pdf_inc).

      /** igc - Nov 4, 2002 - removed now use OS-APPEND
      INPUT STREAM S_pdf_inp FROM VALUE(TT_pdf_image.image_file) BINARY NO-MAP NO-CONVERT UNBUFFERED.
        REPEAT:
          IMPORT STREAM S_pdf_inp UNFORMATTED L_data.
          PUT STREAM S_pdf_inc CONTROL L_data.
        END.
      INPUT STREAM S_pdf_inp CLOSE.
      **/

     OUTPUT STREAM S_pdf_inc CLOSE.
     /* message "here" view-as alert-box. */
     OS-APPEND VALUE(FILE-INFO:FILE-NAME) VALUE(TT_pdf_stream.obj_file ).
     OUTPUT STREAM S_pdf_inc TO VALUE( TT_pdf_stream.obj_file ) UNBUFFERED APPEND.

      /* Get PDF Stream End Offset */
      L_end = SEEK(S_pdf_inc).

      PUT STREAM S_pdf_inc UNFORMATTED
          SKIP(1) "endstream" SKIP
          "endobj" SKIP.

      /* igc - Added Sept 10, 2002 */
      run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "ImageLen",
                            output pdf_inc_ObjectSequence).
      TT_pdf_image.image_len = pdf_inc_ObjectSequence.

      PUT STREAM S_pdf_inc UNFORMATTED
          TT_pdf_image.image_len " 0 obj" SKIP
          "  " (L_end - L_start) SKIP
          "endobj" SKIP.

  END. /* each TT_pdf_image */

END. /* pdf_load_images */

PROCEDURE pdf_load_links:

  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.

  DEFINE VARIABLE L_data  AS RAW NO-UNDO.
  DEFINE VARIABLE L_start AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_end   AS INTEGER NO-UNDO.


  FOR EACH TT_pdf_link WHERE TT_pdf_link.obj_stream = pdfStream:

    /* igc - Added Sept 10, 2002 */
    run ip_ObjectSequence(pdfStream, pdf_inc_ObjectSequence + 1, "Link",
                          output pdf_inc_ObjectSequence).
    TT_pdf_link.link_obj = pdf_inc_ObjectSequence.

    PUT STREAM S_pdf_inc UNFORMATTED
        TT_pdf_link.link_obj " 0 obj" CHR(13)
        "<<" CHR(13)
        "/Type /Annot" CHR(13)
        "/Subtype /Link" CHR(13)
        "/Rect [ "  STRING(TT_pdf_link.link_x) " "
                    STRING(TT_pdf_link.link_y) " "
                    STRING(TT_pdf_link.link_w) " "
                    STRING(TT_pdf_link.link_h) " "
               "]" CHR(13)
         "/Border [0 0 " TT_pdf_link.link_border "]" CHR(13)
         "/H /" TT_pdf_link.link_style CHR(13)
         "/C [ " TT_pdf_link.link_color " ]" CHR(13)
         "/A << " CHR(13)
         "  /Type /Action" CHR(13)
         "  /S /URI" CHR(13)
         "  /URI (" TT_pdf_link.link_Name ")" CHR(13)
         "  >> " CHR(13)
         ">>" CHR(13)
         "endobj" CHR(13).

  END. /* each TT_pdf_link */

END. /* pdf_load_links */

PROCEDURE pdf_get_image_wh:
  DEFINE INPUT PARAMETER pdfStream  AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER Pimage     AS CHARACTER NO-UNDO.

  DEFINE VARIABLE pdf_marker    AS INTEGER NO-UNDO.
  def var         cHexChar1     as char    no-undo.
  def var         lLog1         as log     no-undo.

  INPUT STREAM S_pdf_inp FROM VALUE(Pimage) BINARY NO-MAP NO-CONVERT UNBUFFERED.

    run ip_first_marker(output lLog1).
    IF NOT lLog1 THEN DO:
      RUN pdf_error(pdfStream,"ObjectSequence","Cannot find Stream!").
      RETURN.
    END.


    DO WHILE TRUE:
      run ip_next_marker(output pdf_marker).
      run ip_hex(input pdf_marker, output cHexChar1).
      CASE cHexChar1:
        WHEN {&M_SOF0} OR WHEN {&M_SOF1} OR WHEN {&M_SOF2} OR WHEN {&M_SOF3}
        OR WHEN {&M_SOF5} OR WHEN {&M_SOF6} OR WHEN {&M_SOF7} OR WHEN {&M_SOF9}
        OR WHEN {&M_SOF10} OR WHEN {&M_SOF11} OR WHEN {&M_SOF13}
        OR WHEN {&M_SOF14} OR WHEN {&M_SOF15} THEN DO:
          run ip_process_SOF(output lLog1).
          LEAVE.
        END.
        WHEN {&M_SOS} OR WHEN {&M_EOI} THEN
          LEAVE.
        OTHERWISE do:
          run ip_skip_variable(output lLog1).
        end.
      END CASE.
    END. /* true loop */

  INPUT STREAM S_pdf_inp CLOSE.

END.

PROCEDURE pdf_replace_text:
  DEFINE INPUT-OUTPUT PARAMETER pdfText AS CHARACTER NO-UNDO.

  /* Replace any special characters in the data string since this
     will create a bad PDF doccument */
  pdfText = REPLACE(pdfText,"(","~\(").
  pdfText = REPLACE(pdfText,")","~\)").
  pdfText = REPLACE(pdfText,"[","~\[").
  pdfText = REPLACE(pdfText,"]","~\]").

END. /* pdf_replace_text */

PROCEDURE pdf_reset_all.
  /* clear out all streams, and reset variables as required */

  /* These are the only two variables that don't apear to be reset anywhere */
  ASSIGN pdf_inc_ContentSequence = 0
         pdf_inc_ObjectSequence  = 0.

    FOR EACH TT_pdf_stream:
      DELETE TT_pdf_stream .
    END.

    FOR EACH TT_pdf_param:
      DELETE TT_pdf_param.
    END.

    FOR EACH TT_pdf_error:
      DELETE TT_pdf_error.
    END.

    FOR EACH TT_pdf_object:
      DELETE TT_pdf_object.
    END.

    FOR EACH TT_pdf_content:
      DELETE TT_pdf_content.
    END.

    FOR EACH TT_pdf_info:
      DELETE TT_pdf_info.
    END.

    FOR EACH TT_pdf_image:
      DELETE TT_pdf_image.
    END.

    FOR EACH TT_pdf_font:
      DELETE TT_pdf_font.
    END.
END. /* pdf_reset_all */

PROCEDURE pdf_reset_stream .
  /* Clear out an individual stream - reset the variables */
  DEFINE INPUT PARAMETER pdfStream     AS CHARACTER NO-UNDO.

  /* These are the only two variables that don't apear to be reset anywhere */
  ASSIGN pdf_inc_ContentSequence = 0
         pdf_inc_ObjectSequence  = 0.

  /* As far as I know, you gotta do a for each regardless of version */
  FOR EACH TT_pdf_stream WHERE TT_pdf_stream.obj_stream = pdfStream:
    DELETE TT_pdf_stream .
  END.

  FOR EACH TT_pdf_param WHERE TT_pdf_param.obj_stream = pdfStream:
    DELETE TT_pdf_param.
  END.

  FOR EACH TT_pdf_error WHERE TT_pdf_error.obj_stream = pdfStream:
    DELETE TT_pdf_error.
  END.

  FOR EACH TT_pdf_object WHERE TT_pdf_object.obj_stream = pdfStream:
    DELETE TT_pdf_object.
  END.

  FOR EACH TT_pdf_content WHERE TT_pdf_content.obj_stream = pdfStream:
    DELETE TT_pdf_object.
  END.

  FOR EACH TT_pdf_info WHERE TT_pdf_info.obj_stream = pdfStream:
    DELETE TT_pdf_info.
  END.

  FOR EACH TT_pdf_image WHERE TT_pdf_image.obj_stream = pdfStream:
    DELETE TT_pdf_image.
  END.

  FOR EACH TT_pdf_font WHERE TT_pdf_font.obj_stream = pdfStream:
    DELETE TT_pdf_font.
  END.
END. /* pdf_reset_stream */

PROCEDURE pdf_wrap_text :
/*------------------------------------------------------------------------------
  Purpose:
  Parameters:  <none>
  Notes:
------------------------------------------------------------------------------*/
  DEFINE INPUT PARAMETER pdfStream     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfText       AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFromColumn AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfToColumn   AS INTEGER NO-UNDO.
  DEFINE INPUT PARAMETER pdfAlignMent  AS CHARACTER NO-UNDO.
  DEFINE OUTPUT PARAMETER pdfMaxY      AS INTEGER NO-UNDO.

  DEFINE VAR v-thisline  AS CHAR NO-UNDO.
  DEFINE VAR v-lastline  AS CHAR NO-UNDO.
  DEFINE VAR v-fontsize  AS DECIMAL NO-UNDO.
  DEFINE VAR v-mywidth   AS INTEGER NO-UNDO.
  DEFINE VAR v-originalY AS INTEGER NO-UNDO.
  DEFINE VAR i           AS INTEGER NO-UNDO.
  def    var iWidth      as int     no-undo.

  run ip_pdf_text_width(pdfstream,"i", output v-fontsize).
  ASSIGN v-fontsize = v-fontsize * .81
         v-mywidth = pdfToColumn - pdfFromColumn.
  run ip_pdf_TextY(input pdfstream, output v-originalY).


  ASSIGN pdfText = REPLACE(pdfText,"|","&pipe;").
  /* Spaces */
  ASSIGN pdfText = REPLACE(pdfText," ","| ").
  /* Hyphens */
  ASSIGN pdfText = REPLACE(pdfText,"-","-|").
  /* Commas */
  ASSIGN pdfText = REPLACE(pdfText,",",",|").



  /* Divide up the pdf text into lines of width less than the
     available width */
  DO i = 1 TO NUM-ENTRIES(pdfText,"|"):
    ASSIGN v-lastline = v-thisline.
    ASSIGN v-thisline = v-thisline
                      + REPLACE(ENTRY(i,pdfText,"|"),"&pipe;","|").

    run ip_pdf_text_width(pdfstream,TRIM(v-thisline), output iWidth).

    IF iWidth >= (v-myWidth * v-fontSize)
    THEN DO:
      /* Write out last feasible line */
      ASSIGN v-lastline = TRIM(v-lastline).
      CASE pdfAlignment:
        WHEN "left" THEN
          RUN pdf_text_at(pdfStream,v-lastline,pdfFromColumn).
        WHEN "right" THEN
          RUN pdf_text_to(pdfStream,v-lastline,pdfToColumn).
      END CASE.

      RUN pdf_skip(pdfStream).
      ASSIGN v-thisline = TRIM(REPLACE(ENTRY(i,pdfText,"|"),"&pipe;","|")).
    END.
  END. /* i = NUM-ENTRIES */

  IF v-thisline NE "" THEN
    CASE pdfAlignment:
      WHEN "left" THEN
        RUN pdf_text_at(pdfStream,v-thisline,pdfFromColumn).
      WHEN "right" THEN
        RUN pdf_text_to(pdfStream,v-thisline,pdfToColumn).
    END CASE.

  run ip_pdf_TextY(input pdfstream, output pdfMaxY).

  RUN pdf_set_TextY(pdfStream,v-originalY).

END PROCEDURE. /* pdf_wrap_text */

PROCEDURE pdf_link:
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfFromX    AS INTEGER   NO-UNDO.
  DEFINE INPUT PARAMETER pdfFromY    AS INTEGER   NO-UNDO.
  DEFINE INPUT PARAMETER pdfWidth    AS INTEGER   NO-UNDO.
  DEFINE INPUT PARAMETER pdfHeight   AS INTEGER   NO-UNDO.
  DEFINE INPUT PARAMETER pdfLink     AS CHARACTER NO-UNDO.
  DEFINE INPUT PARAMETER pdfRed      AS DECIMAL   NO-UNDO.
  DEFINE INPUT PARAMETER pdfGreen    AS DECIMAL   NO-UNDO.
  DEFINE INPUT PARAMETER pdfBlue     AS DECIMAL   NO-UNDO.
  DEFINE INPUT PARAMETER pdfBorder   AS INTEGER   NO-UNDO.
  DEFINE INPUT PARAMETER pdfStyle    AS CHARACTER NO-UNDO.

  def var iPage as int no-undo.

  pdfRed   = IF pdfRed < 0 THEN 0
             ELSE IF pdfRed > 1 THEN 1
             ELSE pdfRed.
  pdfGreen = IF pdfGreen < 0 THEN 0
             ELSE IF pdfGreen > 1 THEN 1
             ELSE pdfGreen.
  pdfBlue  = IF pdfBlue < 0 THEN 0
             ELSE IF pdfBlue > 1 THEN 1
             ELSE pdfBlue.

  IF LOOKUP(pdfStyle,"N,I,O,P") = 0 THEN
    pdfStyle = "N".

  run ip_pdf_page(pdfstream, output iPage).
  CREATE TT_pdf_link.
  ASSIGN TT_pdf_link.obj_stream   = pdfStream
         TT_pdf_link.link_name    = pdfLink
         TT_pdf_link.link_page    = iPage
         TT_pdf_link.link_x       = pdfFromX
         TT_pdf_link.link_y       = pdfFromY
         TT_pdf_link.link_h       = pdfHeight
         TT_pdf_link.link_w       = pdfWidth
         TT_pdf_link.link_color   = STRING(pdfRed) + " "
                                  + STRING(pdfGreen) + " "
                                  + STRING(pdfBlue)
         TT_pdf_link.link_border  = IF pdfBorder < 0 THEN 0 ELSE pdfBorder
         TT_pdf_link.link_style   = pdfStyle.

  /***
  CREATE TT_pdf_content.
  ASSIGN TT_pdf_content.obj_stream  = pdfStream.
         TT_pdf_content.obj_seq     = pdf_inc_ContentSequence().
  ASSIGN TT_pdf_content.obj_type    = "LINK"
         TT_pdf_content.obj_content = "/A <<" + CHR(13)
                                    + "/S /URI" + CHR(13)
                                    + "/URI (" + pdfLink + ")" + CHR(13)
                                    + "/Type /Annot" + CHR(13)
                                    + "/SubType /Link" + CHR(13)
                                    + "/Rect ["
                                              + STRING(pdfFromX)
                                              + STRING(pdfFromY)
                                              + STRING(pdfWidth)
                                              + STRING(pdfHeight)
                                              + "]" + CHR(13)
                                    + "/Border [0 0 0]" + CHR(13)
                                    + "/H /I" + CHR(13)
                                    + ">>" + CHR(13)
         TT_pdf_content.obj_line    = pdf_CurrentLine.
         TT_pdf_content.obj_page    = pdf_Page(pdfStream).
  **/
END.

PROCEDURE pdf_new_page :
  DEFINE INPUT PARAMETER pdfStream   AS CHARACTER NO-UNDO.

  def var iIntTextY      as int no-undo.
  def var iPage          as int no-undo.
  def var iPageHeight    as int no-undo.
  def var iTopMargin     as int no-undo.
  def var iBottomMargin  as int no-undo.
  def var iLeftMargin    as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_new_page","Cannot find Stream!").
    RETURN .
  END.

  &IF DEFINED(pdfPageFooter) &THEN
    run ip_pdf_page(pdfstream, output iPage).
    run ip_pdf_BottomMargin(pdfstream, output iBottomMargin).
    run ip_pdf_LeftMargin(pdfstream, output iLeftMargin).
    IF iPage >= 1 THEN DO:
      pdf_ForFooter = TRUE.

      run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
      run ip_pdf_TextY(input pdfstream, output iIntTextY).

      CREATE TT_pdf_content.
      ASSIGN TT_pdf_content.obj_stream  = pdfStream.
             TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
      ASSIGN TT_pdf_content.obj_type    = "TEXT".
             TT_pdf_content.obj_content = "1 0 0 1 " + STRING(iLeftMargin)
                                        + " " + STRING(iBottomMargin) + " Tm".
      ASSIGN TT_pdf_content.obj_line    = iIntTextY.
             TT_pdf_content.obj_page    = iPage.

      {&pdfPageFooter}

      pdf_ForFooter = FALSE.
    END.
  &ENDIF

  RUN pdf_set_Page(pdfStream,(iPage + 1)).
  run ip_pdf_PageHeight(pdfStream,output iPageHeight).
  run ip_pdf_TopMargin(pdfStream,output iTopMargin).
  run ip_pdf_LeftMargin(pdfStream,output iLeftMargin).
  pdf_CurrentLine = iPageHeight - iTopMargin.

  /* Note: the placement of the following commands is important due to the
      setting of the X and Y attributes.  DO NOT CHANGE unless tested
      thoroughly */
  RUN pdf_set_LeftMargin(pdfStream, iLeftMargin).
  RUN pdf_set_TopMargin(pdfStream, iTopMargin).
  run ip_pdf_BottomMargin(pdfStream,output iBottomMargin).
  RUN pdf_set_BottomMargin(pdfStream, iBottomMargin).
  RUN pdf_set_TextX(pdfStream, 0).
  RUN pdf_set_Angle(pdfStream,0).
  RUN pdf_set_TextY(pdfStream, pdf_CurrentLine).
  RUN pdf_set_GraphicX(pdfStream, 0).
  RUN pdf_set_GraphicY(pdfStream, 0).

  &IF DEFINED(pdfPageHeader) &THEN
    run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
    run ip_pdf_TextY(input pdfstream, output iIntTextY).
    run ip_pdf_page(pdfstream, output iPage).

    CREATE TT_pdf_content.
    ASSIGN TT_pdf_content.obj_stream  = pdfStream.
           TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
    ASSIGN TT_pdf_content.obj_type    = "TEXT".
           TT_pdf_content.obj_content = "1 0 0 1 " + STRING(iLeftMargin)
                                      + " " + STRING(pdf_BottomMargin(pdfStream))
                                      + " Tm".
    ASSIGN TT_pdf_content.obj_line    = iIntTextY.
           TT_pdf_content.obj_page    = iPage.

    {&pdfPageHeader}

  &ENDIF

END. /* pdf_new_page */

PROCEDURE pdf_close :
  DEFINE INPUT PARAMETER pdfStream AS CHARACTER NO-UNDO.

  def var iIntTextY     as int no-undo.
  def var iPage         as int no-undo.
  def var iLeftMargin   as int no-undo.
  def var iBottomMargin as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = pdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(pdfStream,"pdf_close","Cannot find Stream!").
    RETURN .
  END.

  DEFINE VARIABLE L_pdf_obj AS INTEGER NO-UNDO.

  FIND FIRST TT_pdf_stream WHERE TT_pdf_stream.obj_stream = pdfStream
       NO-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_stream THEN
    RETURN .

  &IF DEFINED(pdfPageFooter) &THEN
    pdf_ForFooter = TRUE.

    run ip_pdf_inc_ContentSequence(input-output pdf_inc_ContentSequence).
    run ip_pdf_TextY(input pdfstream, output iIntTextY).
    run ip_pdf_LeftMargin(input pdfstream, output iLeftMargin).
    run ip_pdf_BottomMargin(input pdfstream, output iBottomMargin).
    run ip_pdf_page(pdfstream, output iPage).
    CREATE TT_pdf_content.
    ASSIGN TT_pdf_content.obj_stream  = pdfStream.
           TT_pdf_content.obj_seq     = pdf_inc_ContentSequence.
    ASSIGN TT_pdf_content.obj_type    = "TEXT".
           TT_pdf_content.obj_content = "1 0 0 1 " + STRING(iLeftMargin)
                                      + " " + STRING(iBottomMargin) + " Tm".
    ASSIGN TT_pdf_content.obj_line    = iIntTextY.
           TT_pdf_content.obj_page    = iPage.

    {&pdfPageFooter}

    pdf_ForFooter = FALSE.
  &ENDIF

  OUTPUT STREAM S_pdf_inc TO VALUE( TT_pdf_stream.obj_file ) UNBUFFERED.
    /* Output PDF Header Requirements */
    RUN pdf_Header (INPUT TT_pdf_stream.obj_stream).

    /* Load 12 Base Fonts - exclude wingdings etc */
    RUN pdf_Encoding (pdfStream).
    RUN pdf_Fonts (pdfStream).

    IF CAN-FIND (FIRST TT_pdf_error
                 WHERE TT_pdf_error.obj_stream = pdfStream NO-LOCK) THEN DO:
      FOR EACH TT_pdf_error.
        DISPLAY TT_pdf_error.
      END.
    END.
    ELSE DO:
      /* Load Embedded Fonts */
      IF CAN-FIND( FIRST TT_pdf_font WHERE TT_pdf_font.obj_stream = pdfStream
                                       AND TT_pdf_font.font_file <> "PDFBASE14" NO-LOCK)
      THEN RUN pdf_Load_Fonts (pdfStream).

      /* Load Embedded Images */
      IF CAN-FIND( FIRST TT_pdf_image NO-LOCK)
      THEN RUN pdf_Load_Images (pdfStream).

      /* Load Annotations */
      IF CAN-FIND( FIRST TT_pdf_link NO-LOCK)
      THEN RUN pdf_Load_Links (pdfStream).

      RUN pdf_Resources (pdfStream).
      RUN pdf_Content (pdfStream).
      RUN pdf_Xref (pdfStream).
    END.

  OUTPUT STREAM S_pdf_inc CLOSE.

END PROCEDURE. /* pdf_Close */

procedure ip_hex:
  def input  parameter icInt  as int  no-undo.
  def output parameter ocChar as char no-undo.

  DEF VAR j AS INT  NO-UNDO.
  DEF VAR h AS CHAR NO-UNDO.
  DO WHILE TRUE:
    j = icInt MODULO 16.
    h = (IF j < 10 THEN STRING(j) ELSE CHR(ASC("A") + j - 10)) + h.
    IF icInt < 16 THEN LEAVE.
      icInt = (icInt - j) / 16.
    END.
  ocChar = ("0x" + h).
END procedure. /* hex */

procedure ip_nextbyte:
  def output parameter oiInt as int no-undo.

  DEFINE VARIABLE L_data    AS RAW NO-UNDO.

  LENGTH(L_data) = 1.

  IMPORT STREAM S_pdf_inp UNFORMATTED L_data.
  oiInt = GET-BYTE(L_data,1).

END procedure. /* nextbyte */

procedure ip_next2bytes:

  def output parameter oiInt as int no-undo.

  DEFINE VARIABLE L_c1      AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_c2      AS INTEGER NO-UNDO.

  run ip_nextbyte(output L_c1).
  run ip_nextbyte(output L_c2).

  oiInt = INT(L_c1 * EXP(2, 8) + L_c2).

END procedure. /* next2bytes */

procedure ip_first_marker:

  def output parameter olLog as log no-undo.

  DEFINE VARIABLE L_c1        AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_c2        AS INTEGER NO-UNDO.

  def var cHexChar1 as char no-undo.
  def var cHexChar2 as char no-undo.

  run ip_nextbyte(output L_c1).
  run ip_nextbyte(output L_c2).

  run ip_hex(input L_c1,output cHexChar1).
  run ip_hex(input L_c2,output cHexChar2).

  IF cHexChar1 <> {&M_Mark} AND cHexChar2 <> {&M_SOI} THEN
    olLog = FALSE.
  ELSE olLog = TRUE.
END procedure. /* first_marker */

procedure ip_next_marker:

  def output parameter oiInt as int no-undo.

  DEFINE VARIABLE L_data    AS RAW NO-UNDO.

  def var cHexChar1 as char no-undo.

  LENGTH(L_data) = 1.

  DEFINE VARIABLE L_c       AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_discard AS INTEGER NO-UNDO.

  run ip_nextbyte(output L_c).

  run ip_hex(input L_c, output cHexChar1).

  DO WHILE cHexChar1 <> {&M_MARK}:
    L_discard = L_discard + 1.
    run ip_nextbyte(output L_c).
    run ip_hex(input L_c, output cHexChar1).
  END. /* <> 0xFF */

  run ip_hex(input L_c, output cHexChar1).
  DO WHILE cHexChar1 = {&M_MARK}:
    run ip_nextbyte(output L_c).
    run ip_hex(input L_c, output cHexChar1).
  END.

  oiInt = L_c.
END procedure. /* next_marker */

procedure ip_skip_variable:

  def output parameter olLog as log no-undo.

  DEFINE VARIABLE L_Length  AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_Loop    AS INTEGER NO-UNDO.

  def    var      iInt      as int     no-undo.


  run ip_next2bytes(output L_length).

  DO L_Loop = 1 TO (L_Length - 2):
    run ip_nextbyte(output iInt).
  END. /* Loop */
  olLog = true.
END procedure. /* skip_variable */

procedure ip_process_SOF:

  def output parameter olLog as log no-undo.

  DEFINE VARIABLE L_Length  AS INTEGER NO-UNDO.
  def    var      iInt      as int     no-undo.

  run ip_next2bytes(output iInt).       /* Skip Length */
  run ip_nextbyte(output iInt).         /* Skip Data Precision */
  run ip_next2bytes(output pdf_height).
  run ip_next2bytes(output pdf_width).

  olLog = true.

END procedure. /* process_SOF */

procedure ip_pdf_inc_ContentSequence:

  def input-output parameter iiInt as int no-undo.

  iiInt = iiInt + 1.

end procedure. /* pdf_inc_ContentSequence */

procedure ip_pdf_Font:

  def input  parameter icChar as char no-undo.
  def output parameter ocChar as char no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icChar NO-LOCK)
  THEN DO:
    RUN pdf_error(icChar,"pdf_Font","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND TT_pdf_param WHERE TT_pdf_param.obj_stream    = icChar
                      AND TT_pdf_param.obj_parameter = "Font"
                      NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ocChar = TT_pdf_param.obj_value.
  ELSE
    ocChar = "Courier".

END procedure. /* pdf_Font */

procedure ip_pdf_TextX:

  def input  parameter icChar as char no-undo.
  def output parameter oiInt  as int  no-undo.

  IF NOT CAN-FIND (FIRST TT_pdf_stream
                   WHERE TT_pdf_stream.obj_stream = icChar NO-LOCK)
  THEN DO:
    RUN pdf_error(icChar,"pdf_TextX","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icChar
                            AND TT_pdf_param.obj_parameter = "TextX"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    oiInt = INT(TT_pdf_param.obj_value).
  ELSE
    oiInt = 0.

END procedure. /* pdf_TextX*/

procedure ip_pdf_TextY:

  def input  parameter icChar as char no-undo.
  def output parameter oiInt  as int  no-undo.

  FIND FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icChar NO-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_stream THEN DO:
    RUN pdf_error(icChar,"pdf_TextY","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icChar
                            AND TT_pdf_param.obj_parameter = "TextY"
                            NO-LOCK NO-ERROR.

  IF AVAIL TT_pdf_param THEN
    oiInt = INT(TT_pdf_param.obj_value).
  ELSE
    oiInt = 0.

END procedure. /* pdf_TextY */

procedure ip_ObjectSequence:

  def  input parameter icpdfStream     as char no-undo.
  def  input parameter iipdfSequence   as int  no-undo.
  def  input parameter icpdfObjectDesc as char no-undo.
  def output parameter oiInt as int no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"ObjectSequence","Cannot find Stream!").
    RETURN ERROR.
  END.

  CREATE TT_pdf_object.
  ASSIGN TT_pdf_object.obj_stream = icpdfStream
         TT_pdf_object.obj_nbr    = iipdfSequence
         TT_pdf_object.obj_desc   = icpdfObjectDesc
         TT_pdf_object.obj_offset = IF iipdfSequence <> 0 THEN
                                      SEEK(S_pdf_inc) + 1
                                    ELSE 0
         TT_pdf_object.gen_nbr   = IF iipdfSequence <> 0 THEN 0
                                   ELSE 65535
         TT_pdf_object.obj_type  = IF iipdfSequence = 0 THEN "f"
                                   ELSE "n".

  pdf_inc_ObjectSequence = iipdfSequence.

  oiInt = pdf_inc_ObjectSequence.

END procedure. /* ObjectSequence */

procedure ip_pdf_VerticalSpace:

  def input  parameter icpdfStream as char no-undo.
  def output parameter oiInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_VerticalSpace","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "VerticalSpace"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    oiInt = INT(TT_pdf_param.obj_value).
  ELSE
    oiInt = 10.

END procedure. /* pdf_VerticalSpace */

procedure ip_pdf_PointSize:

  def input  parameter icpdfStream as char no-undo.
  def output parameter odDec       as dec  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_PointSize","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "PointSize"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    odDec = DEC(TT_pdf_param.obj_value).
  ELSE
    odDec = 10.0.

END procedure. /* pdf_PointSize */

procedure ip_pdf_text_width:

  def input  parameter icpdfStream as char no-undo.
  def input  parameter icpdfText   as char no-undo.

  def output parameter iiWidth     as int  no-undo.

  DEFINE VARIABLE L_width  AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_font    AS CHARACTER NO-UNDO.

  def var dpdfPointSize as dec no-undo.

  run ip_pdf_Font(input icpdfstream, output L_font).

  DEFINE VARIABLE L_Loop    AS INTEGER NO-UNDO.
  DEFINE VARIABLE L_tot     AS INTEGER NO-UNDO.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_text_width","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_font WHERE TT_pdf_font.obj_stream = icpdfStream
                           AND TT_pdf_font.font_name  = L_Font
                           NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_font THEN DO:
    run ip_pdf_PointSize(icpdfStream, output dpdfPointSize).
    IF TT_pdf_Font.font_type = "FIXED" THEN
      L_width = INT((LENGTH(icpdfText) * INT(TT_pdf_font.font_width) / 1000)
                    * dpdfPointSize).

    ELSE DO:
      DO L_loop = 1 TO LENGTH(icpdfText):
        L_tot = L_tot + INT(ENTRY(ASC(SUBSTR(icpdfText,L_Loop,1)) ,
                                  TT_pdf_Font.font_width, " ")) NO-ERROR.
      END.
      run ip_pdf_PointSize(icpdfStream, output dpdfPointSize).
      l_width = INT((L_tot / 1000) * dpdfPointSize).
    END. /* Variable Width Font */
  END. /* Found the current font */

  iiWidth = l_width.
END procedure. /* pdf_text_color */

procedure ip_pdf_TextRed:

  def input  parameter icpdfstream as char no-undo.
  def output parameter odRed       as dec  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_TextRed","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "TextRed"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    odRed = DEC(TT_pdf_param.obj_value).
  ELSE
    odRed = 0.0.

END procedure. /* pdf_TextRed */

procedure ip_pdf_TextGreen:

  def input  parameter icpdfstream as char no-undo.
  def output parameter odGreen     as dec  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_TextGreen","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "TextGreen"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    odGreen = DEC(TT_pdf_param.obj_value).
  ELSE
    odGreen = 0.0.

END procedure. /* pdf_TextGreen */

procedure ip_pdf_TextBlue:

  def input  parameter icpdfstream as char no-undo.
  def output parameter odBlue      as dec  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_TextGreen","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "TextBlue"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    odBlue = DEC(TT_pdf_param.obj_value).
  ELSE
    odBlue = 0.0.

END procedure. /* pdf_TextGreen */

procedure ip_pdf_Page:

  def input parameter  icpdfstream as char no-undo.
  def output parameter oiPage      as int  no-undo.

  FIND FIRST TT_pdf_stream
       WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK NO-ERROR.
  IF NOT AVAIL TT_pdf_stream THEN DO:
    RUN pdf_error(icpdfStream,"pdf_Page","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "Page"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    oiPage = INT(TT_pdf_param.obj_value).
  ELSE
    oiPage = 0.

END procedure. /* pdf_Page */

procedure ip_pdf_PageWidth:

  def input  parameter icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_PageWidth","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "PageWidth"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ioInt = INT(TT_pdf_param.obj_value).
  ELSE
    ioInt = 612.

END procedure. /* pdf_PageWidth */

procedure ip_pdf_Pageheight:

  def input  parameter icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"PageHeight","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "PageHeight"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ioInt = INT(TT_pdf_param.obj_value).
  ELSE
    ioInt = 792.

END procedure. /* pdf_PageHeight */

procedure ip_pdf_Angle:

  def input  parameter icpdfStream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_Angle","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "Angle"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN ioInt = INT(TT_pdf_param.obj_value).
  ELSE ioInt = ?.

END. /* pdf_Angle */

procedure ip_pdf_TopMargin:

  def input  parameter icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_TopMargin","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "TopMargin"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ioInt = INT(TT_pdf_param.obj_value).
  ELSE
    ioInt = 50.

END procedure. /* pdf_TopMargin */

procedure ip_pdf_BottomMargin:

  def input  parameter icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_BottomMargin","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "BottomMargin"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ioInt = INT(TT_pdf_param.obj_value).
  ELSE
    ioInt = 50.

END procedure. /* pdf_BottomMargin */

procedure ip_pdf_GraphicX:
  def input  parameter icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_GraphicX","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "GraphicX"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ioInt = INT(TT_pdf_param.obj_value).
  ELSE
    ioInt = 0.

END procedure. /* pdf_GraphicX */

procedure ip_pdf_GraphicY:

  def input  parameter icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_GraphicY","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "GraphicY"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ioInt = INT(TT_pdf_param.obj_value).
  ELSE
    ioInt = 0.

end procedure. /* pdf_GraphicY */

procedure ip_pdf_get_info:

  def input  parameter icpdfstream    as char no-undo.
  def input  parameter icpdfAttribute as char no-undo.
  def output parameter coChar         as char no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_get_info","Cannot find Stream!").
    RETURN ERROR.
  END.

  DEFINE VARIABLE L_option  AS CHARACTER NO-UNDO.
  L_Option = "Author,Creator,Producer,Keywords,Subject,Title".

  IF LOOKUP(icpdfAttribute,L_option) = 0 THEN DO:
    RUN pdf_error(icpdfStream,"ObjectSequence","Invalid Attribute passed!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_info WHERE TT_pdf_info.obj_stream = icpdfStream
                           AND TT_pdf_info.info_attr  = icpdfAttribute
                           NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_info THEN
    coChar = TT_pdf_info.info_value.

END procedure. /* pdf_get_info */

procedure ip_pdf_LeftMargin:
  def input  parameter icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_LeftMargin","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "LeftMargin"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN
    ioInt = INT(TT_pdf_param.obj_value).
  ELSE
    ioInt = 10.

END procedure. /* pdf_LeftMargin */

procedure ip_pdf_get_numFittingChars:

  def input  parameter icpdfstream      as char no-undo.
  def input  parameter icpdfText        as char no-undo.
  def input  parameter pdfRequiredWidth as int  no-undo.
  def output parameter ioInt            as int  no-undo.

  DEFINE VARIABLE L_font        AS CHARACTER NO-UNDO.
  DEFINE VARIABLE L_tot         AS INTEGER   NO-UNDO.
  def    var      dpdfPointSize as dec       no-undo.

  DEFINE VARIABLE iReqFMWidth AS INTEGER NO-UNDO. /* Required font-metric width */
  DEFINE VARIABLE iCurFMWidth AS INTEGER NO-UNDO. /* Font-metric width of chars that fit */
  DEFINE VARIABLE iCurChar    AS INTEGER NO-UNDO. /* Char index up to */

  run ip_pdf_Font(input icpdfstream, output L_font).
  run ip_pdf_PointSize(icpdfStream, output dpdfPointSize).
  iReqFMWidth  = (pdfRequiredWidth / dpdfPointSize ) * 1000.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_text_width","Cannot find Stream!").
    RETURN ERROR.
  END.

  iCurChar = 0.
  iCurFMWidth = 0.
  FIND FIRST TT_pdf_font WHERE TT_pdf_font.obj_stream = icpdfStream
                           AND TT_pdf_font.font_name  = L_Font
                           NO-LOCK NO-ERROR.
  IF AVAILABLE TT_pdf_font THEN
  DO iCurChar = 1 TO LENGTH(icpdfText):

    /* Keep looping until width is too wide, then return one less char */
    iCurFMWidth = iCurFMWidth +
                   (IF TT_pdf_Font.font_type = "FIXED" THEN
                      INTEGER(TT_pdf_font.font_width)
                    ELSE
                      INTEGER(ENTRY(ASC(SUBSTR(icpdfText, iCurChar, 1)),  /* ASCII Value of char at cur pos */
                              TT_pdf_Font.font_width,             /* Space-seperated values list widths */
                              " ")) ) NO-ERROR.
    IF iCurFMWidth > iReqFMWidth THEN
    do:
      ioInt = iCurChar - 1.
      return.
    end.
  END. /* Loop through text */

  ioInt = iCurChar.
END procedure. /* pdf_get_numFittingChars */

procedure ip_pdf_Orientation:

  def input  parameter icpdfstream as char no-undo.
  def output parameter coChar      as char no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_Orientation","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "Orientation"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN coChar = TT_pdf_param.obj_value.
  ELSE coChar = "Portrait".

end procedure. /* pdf_Orientation */

procedure ip_pdf_PaperType:

  def input  parameter icpdfstream as char no-undo.
  def output parameter coChar      as char no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"icpdf_PaperType","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "PaperType"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN coChar = TT_pdf_param.obj_value.
  ELSE coChar = "LETTER".

end procedure. /* pdf_PaperType */

procedure ip_pdf_Render:

  def input parameter  icpdfstream as char no-undo.
  def output parameter ioInt       as int  no-undo.

  IF NOT CAN-FIND(FIRST TT_pdf_stream
                  WHERE TT_pdf_stream.obj_stream = icpdfStream NO-LOCK)
  THEN DO:
    RUN pdf_error(icpdfStream,"pdf_Render","Cannot find Stream!").
    RETURN ERROR.
  END.

  FIND FIRST TT_pdf_param WHERE TT_pdf_param.obj_stream    = icpdfStream
                            AND TT_pdf_param.obj_parameter = "Render"
                            NO-LOCK NO-ERROR.
  IF AVAIL TT_pdf_param THEN ioInt = INT(TT_pdf_param.obj_value).
  ELSE ioInt = 0.

end procedure. /* pdf_Render */

procedure ip_pdf_get_wrap_length:

  def input  parameter icpdfstream as char no-undo.
  def input  parameter icpdftext   as char no-undo.
  def input  parameter iipdfwidth  as int  no-undo.
  def output parameter ioInt       as int  no-undo.

  DEFINE VAR pdfHeight  AS INTEGER NO-UNDO.

  DEFINE VAR v-thisline  AS CHAR NO-UNDO.
  DEFINE VAR v-lastline  AS CHAR NO-UNDO.
  DEFINE VAR v-fontsize  AS DECIMAL NO-UNDO.
  DEFINE VAR i           AS INTEGER NO-UNDO.
  DEFINE VAR v-pointsize AS INTEGER NO-UNDO.
  def    var iWidth      as int     no-undo.

  run ip_pdf_text_width (icpdfStream, icpdfText, output v-fontsize).

  run ip_pdf_PointSize(icpdfStream, output v-pointSize).

  ASSIGN icpdfText = REPLACE(icpdfText,"|","&pipe;").
  /* Spaces */
  ASSIGN icpdfText = REPLACE(icpdfText," ","| ").
  /* Hyphens */
  ASSIGN icpdfText = REPLACE(icpdfText,"-","-|").
  /* Commas */
  ASSIGN icpdfText = REPLACE(icpdfText,",",",|").

  /* Divide up the pdf text into lines of width less than the
     available width */
  DO i = 1 TO NUM-ENTRIES(icpdfText,"|"):
    ASSIGN v-lastline = v-thisline.
    ASSIGN v-thisline = v-thisline
                      + REPLACE(ENTRY(i,icpdfText,"|"),"&pipe;","|").

    run ip_pdf_text_width (icpdfStream, TRIM(v-thisline), output iWidth).
    IF iWidth >= (iipdfWidth * v-fontSize)
    THEN DO:
        ASSIGN pdfHeight = pdfHeight + (v-pointsize * 2).
        ASSIGN v-thisline = TRIM(REPLACE(ENTRY(i,icpdfText,"|"),"&pipe;","|")).
    END.
  END.

  IF v-thisline NE "" THEN
    ASSIGN pdfHeight = pdfHeight + v-pointsize.

  ioInt = pdfHeight.
END procedure. /* pdf_get_wrap_length */

procedure pdf_TotalPages:

  def input  parameter icpdfstream as char no-undo.
  def output parameter coChar      as char no-undo.

  coChar = "@@TOTALPages-Spdf".

END procedure. /* pdf_TotalPages */

/* end of XXPDF_INC.I */
